<?php

namespace Drupal\ai_story_teller\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Profile form.
 */
final class ProfileForm extends EntityForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new ProfileForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\ai_story_teller\Entity\Profile $profile */
    $profile = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $profile->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('ID'),
      '#default_value' => $profile->id(),
      '#machine_name' => [
        'exists' => [$this, 'exists'],
      ],
      '#disabled' => !$profile->isNew(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $profile->status(),
    ];

    $form['prompt_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt Template'),
      '#default_value' => $profile->get('prompt_template'),
    ];

    // provider and model settings
    $form['provider_settings'] = [
      '#type' => 'table',
      '#title' => $this->t('AI Provider and Model Settings'),
      '#header' => [
        $this->t('Provider'),
        $this->t('Model'),
        $this->t('Settings'),
      ],
      '#prefix' => '<div id="provider-settings-wrapper">',
      '#suffix' => '</div>',
      '#empty' => $this->t('No configuration added yet.'),
    ];

    $provider_settings = $profile->getProviderSettings();

    $form['provider_settings'][0]['provider'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Provider'),
      '#default_value' => $provider_settings ? $provider_settings->provider : '',
      '#required' => TRUE,
    ];

    $form['provider_settings'][0]['model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model'),
      '#default_value' => $provider_settings ? $provider_settings->model : '',
      '#required' => TRUE,
    ];

    $form['provider_settings'][0]['model_settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Model Settings'),
      '#default_value' => $provider_settings ? $provider_settings->model_settings : '',
      '#required' => TRUE,
    ];
    // end of provider and model settings

    // reports storage
    $reports_storage_values = $profile->get('reports_storage');
    $title_options = [];
    $body_options = [];
    if (!empty($reports_storage_values) && is_string($reports_storage_values)) {
      $reports_storage_values = json_decode($reports_storage_values);
      $title_options = $this->getFieldsOptions($reports_storage_values->content_type);
      $body_options = $this->getFieldsOptions($reports_storage_values->content_type);
    }
    else {
      $reports_storage_values = NULL;
    }

    $form['reports_storage'] = [
      '#type' => 'table',
      '#title' => $this->t('Reports Storage'),
      '#header' => [
        $this->t('Reports Storage'),
        $this->t('Title Field'),
        $this->t('Body Field'),
      ],
      '#prefix' => '<div id="reports-storage-wrapper">',
      '#suffix' => '</div>',
      '#empty' => $this->t('No configuration added yet.'),
    ];

    $form['reports_storage'][0]['content_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Type'),
      '#options' => $this->getContentTypeOptions(),
      '#default_value' => $reports_storage_values->content_type,
      '#ajax' => [
        'callback' => '::updateFieldsAjax',
        'wrapper' => 'reports-storage-wrapper',
        'event' => 'change',
      ],
    ];

    $content_type = $form_state->getValue([
      'reports_storage',
      0,
      'content_type',
    ]);

    if ($content_type) {
      $title_options = $this->getFieldsOptions($content_type);
      $body_options = $this->getFieldsOptions($content_type);
    }

    $form['reports_storage'][0]['title_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Title Field'),
      '#options' => $title_options,
      '#default_value' => $reports_storage_values->title_field,
    ];

    $form['reports_storage'][0]['body_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Body Field'),
      '#options' => $body_options,
      '#default_value' => $reports_storage_values->body_field,
    ];
    // end of reports storage

    // Load source entities.
    $sources = $this->entityTypeManager->getStorage('source')->loadMultiple();
    $source_options = [];
    foreach ($sources as $source) {
      $source_options[$source->id()] = $source->label();
    }

    // Initialize the sources form.
    if ($form_state->get('sources_field') === NULL) {
      $sources_field = $profile->get('sources') ?: [
        [
          'source_id' => '',
          'weight' => 0,
        ],
      ];
      $form_state->set('sources_field', $sources_field);
    }
    else {
      $sources_field = $form_state->get('sources_field');
    }

    $form['sources'] = [
      '#type' => 'table',
      '#title' => $this->t('Sources'),
      '#header' => [
        $this->t('Source'),
        $this->t('Weight'),
        $this->t('Remove'),
      ],
      '#prefix' => '<div id="sources-wrapper">',
      '#suffix' => '</div>',
      '#empty' => $this->t('No sources added yet.'),
    ];

    foreach ($sources_field as $delta => $source) {
      $form['sources'][$delta]['source_id'] = [
        '#type' => 'select',
        '#title' => $this->t('Source'),
        '#title_display' => 'invisible',
        '#options' => $source_options,
        '#default_value' => $source['source_id'] ?? '',
        '#required' => TRUE,
      ];

      $form['sources'][$delta]['weight'] = [
        '#type' => 'number',
        '#title' => $this->t('Weight'),
        '#title_display' => 'invisible',
        '#default_value' => $source['weight'] ?? 0,
        '#required' => TRUE,
      ];

      $form['sources'][$delta]['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#submit' => ['::removeSource'],
        '#name' => 'remove_source_' . $delta,
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => '::addSourceAjax',
          'wrapper' => 'sources-wrapper',
        ],
      ];
    }

    $form['sources']['actions'] = [
      '#type' => 'actions',
    ];

    $form['sources']['actions']['add_source'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Source'),
      '#submit' => ['::addSource'],
      '#ajax' => [
        'callback' => '::addSourceAjax',
        'wrapper' => 'sources-wrapper',
      ],
    ];

    return $form;
  }

  private function getContentTypeOptions() {
    $content_types = NodeType::loadMultiple();
    $ctype_options = [];
    foreach ($content_types as $type) {
      $ctype_options[$type->id()] = $type->label();
    }
    return $ctype_options;
  }

  private function getFieldsOptions($content_type) {
    $fields_options = [
      'title' => $this->t('Title'),
    ];

    $fields = $this->entityTypeManager->getStorage('field_config')
      ->loadByProperties(['entity_type' => 'node', 'bundle' => $content_type]);
    foreach ($fields as $field) {
      if (in_array($field->get('field_type'), [
        'string',
        'text',
        'text_long',
        'text_with_summary',
      ])) {
        $fields_options[$field->getName()] = $field->getLabel();
      }
    }
    return $fields_options;
  }

  /**
   * AJAX callback handler for updating fields.
   */
  public function updateFieldsAjax(array &$form, FormStateInterface $form_state) {
    return $form['reports_storage'];
  }

  /**
   * AJAX callback handler.
   */
  public function addSourceAjax(array &$form, FormStateInterface $form_state) {
    return $form['sources'];
  }

  /**
   * Submit handler for adding a source.
   */
  public function addSource(array &$form, FormStateInterface $form_state) {
    $sources_field = $form_state->get('sources_field');
    $sources_field[] = ['source_id' => '', 'weight' => count($sources_field)];
    $form_state->set('sources_field', $sources_field);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing a source.
   */
  public function removeSource(array &$form, FormStateInterface $form_state) {
    $sources_field = $form_state->get('sources_field');
    $triggering_element = $form_state->getTriggeringElement();
    $source_index = array_reverse(explode('_', $triggering_element['#name']))[0];
    if (isset($sources_field[$source_index])) {
      unset($sources_field[$source_index]);
    }
    $form_state->set('sources_field', array_values($sources_field));
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\ai_story_teller\Entity\Profile $profile */
    $profile = $this->entity;

    // Set the sources from the form input.
    $sources_field = $form_state->getValues()['sources'];
    $sources = [];
    foreach ($sources_field as $source) {
      $sources[] = $source;
    }
    $profile->set('sources', array_filter($sources));

    $report_storage_settings = [];
    $report_storage_settings['content_type'] = $form_state->getValue('reports_storage')[0]['content_type'];
    $report_storage_settings['title_field'] = $form_state->getValue('reports_storage')[0]['title_field'];
    $report_storage_settings['body_field'] = $form_state->getValue('reports_storage')[0]['body_field'];
    $profile->set('reports_storage', json_encode($report_storage_settings));

    $provider_settings = [];
    $provider_settings['provider'] = $form_state->getValue('provider_settings')[0]['provider'];
    $provider_settings['model'] = $form_state->getValue('provider_settings')[0]['model'];
    $provider_settings['model_settings'] = $form_state->getValue('provider_settings')[0]['model_settings'];
    $profile->set('provider_settings', json_encode($provider_settings));

    $status = $profile->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label Profile.', [
          '%label' => $profile->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label Profile.', [
          '%label' => $profile->label(),
        ]));
    }

    $form_state->setRedirectUrl($profile->toUrl('collection'));
  }

  /**
   * Check if the ID already exists.
   */
  public function exists($id) {
    return (bool) $this->entityTypeManager->getStorage('profile')->load($id);
  }

}
