<?php

declare(strict_types=1);

namespace Drupal\ai_story_teller\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai_story_teller\Entity\Source;

/**
 * Source form.
 */
final class SourceForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\ai_story_teller\Entity\Source $source */
    $source = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $source->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $source->id(),
      '#machine_name' => [
        'exists' => [Source::class, 'load'],
      ],
      '#disabled' => !$source->isNew(),
    ];

    $form['url'] = [
      '#type' => 'url',
      '#title' => $this->t('URL'),
      '#default_value' => $source->get('url'),
      '#required' => TRUE,
    ];

    // Get the list of available plugins.
    $plugin_manager = \Drupal::service('ai_reporter.scraper_plugin_manager');
    $plugin_definitions = $plugin_manager->getDefinitions();
    $plugin_options = [];
    foreach ($plugin_definitions as $plugin_id => $plugin_definition) {
      $plugin_options[$plugin_id] = $plugin_definition['label'];
    }

    $form['scraper_plugin'] = [
      '#type' => 'select',
      '#title' => $this->t('Scraper Plugin'),
      '#options' => $plugin_options,
      '#default_value' => $source->get('scraper_plugin'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new source %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated source %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

}
