<?php

declare(strict_types=1);

namespace Drupal\ai_search_block_log\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for an ai search block log entity type.
 */
final class AISearchBlockLogSettingsForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_search_block_log_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['settings'] = [
      '#markup' => $this->t('Settings form for an ai search block log entity type.'),
    ];
    $expiry = $this->configFactory()
      ->get('ai_search_block_log.settings')
      ->get('expiry');
    $form['expiration'] = [
      '#type' => 'select',
      '#title' => $this->t('Expiration'),
      '#default_value' => $expiry ?? 'year',
      '#description' => $this->t('This is the amount of time the system will keep the logs'),
      '#options' => [
        'day' => $this->t('1 day'),
        'week' => $this->t('1 week'),
        'month' => $this->t('1 month'),
        'year' => $this->t('1 year'),
      ],
    ];

    $default_prompt = 'INSTRUCTIONS
------------
You are summarising data for a website ai search manager.
You write professional english. You do not answer a question.
I will give you a number of questions, answers with scores, and user feedback. These are searches that received low scores (1-2) or had user feedback provided.
From the following questions, answers, and feedback:
- find recurring themes in the questions, answers, or feedback
- identify common problems or issues
- explain the recurring themes very briefly. You get to write max 5 sentences. Make the sentences simple and short. Easy to understand.

RESPONSE FORMAT
-----------
You may use <i> <u> <b> <ul><li> html tags in your response.
If you find no questions or responses just say: "No data to analyse."

[QUESTIONS AND ANSWERS WITH SCORES]
-----------
[questions]';

    $ai_analysis_prompt = $this->configFactory()
      ->get('ai_search_block_log.settings')
      ->get('ai_analysis_prompt');

    $form['ai_analysis_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('AI Analysis Prompt'),
      '#default_value' => $ai_analysis_prompt ?? $default_prompt,
      '#description' => $this->t('The prompt used for AI analysis of low-rated searches and feedback. Use [questions] as a placeholder that will be replaced with questions, answers, scores, and feedback.'),
      '#rows' => 15,
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $this->configFactory()
      ->getEditable('ai_search_block_log.settings')
      ->set('expiry', $values['expiration'])
      ->set('ai_analysis_prompt', $values['ai_analysis_prompt'])
      ->save();
    $this->messenger()
      ->addStatus($this->t('The configuration has been updated.'));
  }

}
