<?php

declare(strict_types=1);

namespace Drupal\ai_search_block\Controller;

use Drupal\ai_search_block\AiSearchBlockHelper;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * An example controller.
 */
class AiSearchBlockController extends ControllerBase {

  /**
   * The block storage.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $blockStorage;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructor.
   *
   * @param \Drupal\ai_search_block\AiSearchBlockHelper $searchBlockHelper
   *   The search block helper.
   * @param mixed $blockStorage
   *   The block storage.
   * @param mixed $requestStack
   *   The request stack.
   * @param mixed $renderer
   *   The renderer service.
   */
  public function __construct(
    protected AiSearchBlockHelper $searchBlockHelper,
    mixed $blockStorage,
    mixed $requestStack,
    mixed $renderer,
  ) {
    $this->blockStorage = $blockStorage;
    $this->requestStack = $requestStack;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): AiSearchBlockController {
    return new static(
      $container->get('ai_search_block.helper'),
      $container->get('entity_type.manager')->getStorage('block'),
      $container->get('request_stack'),
      $container->get('renderer')
    );
  }

  /**
   * Returns a renderable array for a test page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|string|\Symfony\Component\HttpFoundation\StreamedResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function search(Request $request): JsonResponse|string|StreamedResponse {
    if ($request->request->get('block_id')) {
      $query = $request->request->get('query');
      $block_id = $request->request->get('block_id');
      $stream = $request->request->get('stream');
    }
    else {
      $input = file_get_contents('php://input');
      $data = Json::decode($input ?: '{}');
      $query = $data['query'] ?? NULL;
      $stream = $data['stream'] ?? NULL;
      $block_id = $data['block_id'] ?? NULL;
    }
    if (empty($block_id)) {
      return new JsonResponse(
        [
          'response' => 'Missing required parameter: block_id',
          'status' => 'error',
        ],
        400
      );
    }
    /** @var \Drupal\block\BlockInterface|null $block */
    $block = $this->blockStorage->load($block_id);
    $logId = 0;
    if (function_exists('ai_search_block_log_start')) {
      $logId = (int) ai_search_block_log_start($block_id, (int) $this->currentUser()->id(),
        $query);
    }
    if ($block) {
      $settings = $block->get('settings');
      $this->searchBlockHelper->setConfig($settings);
      $this->searchBlockHelper->setBlockId($block_id);
      $this->searchBlockHelper->logId = $logId;
      $results = $this->searchBlockHelper->searchRagAction($query);

      if ($stream === 'true' || $stream === 'TRUE') {
        header('X-Accel-Buffering: no');
        // Making maximum execution time unlimited.
        set_time_limit(0);
        ob_implicit_flush();

        return $results;
      }

      return $results;
    }

    if (function_exists('ai_search_block_log_add_response')) {
      ai_search_block_log_add_response($logId, 'There was an error fetching your data');
    }

    return new JsonResponse([
      'response' => 'There was an error fetching your data',
      'log_id' => $logId,
    ]);
  }

  /**
   * Get database results from a view.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with rendered HTML.
   */
  public function getDbResults(Request $request): JsonResponse {
    // Zero-based, ensure non-negative.
    $query    = trim((string) $request->request->get('query'));
    $block_id = $request->request->get('block_id');
    $page     = max(0, (int) ($request->request->get('page') ?? 0));

    if (empty($block_id)) {
      return new JsonResponse(['html' => '<p>Error: Missing block_id.</p>'], 400);
    }

    /** @var \Drupal\block\BlockInterface|null $block */
    $block = $this->blockStorage->load($block_id);
    if (!$block) {
      return new JsonResponse(['html' => '<p>Error: Invalid block configuration.</p>'], 400);
    }

    $settings = $block->get('settings');
    if (empty($settings['database_results_view'])) {
      return new JsonResponse(['html' => '<p>No database view configured.</p>'], 400);
    }

    $view_parts = explode(':', $settings['database_results_view']);
    if (count($view_parts) !== 2) {
      return new JsonResponse([
        'html' => '<p>Error: Invalid view configuration format.</p>',
      ], 400);
    }

    [$view_id, $display_id] = $view_parts;
    $view = Views::getView($view_id);
    if (!$view) {
      return new JsonResponse(['html' => '<p>Error: Could not load view.</p>'], 400);
    }

    $view->setDisplay($display_id);

    // Exposed input (optional).
    $filters = $view->display_handler->getOption('filters') ?: [];
    $filter_key = 'search_api_fulltext';
    foreach ($filters as $filter) {
      if (!empty($filter['expose']['identifier']) &&
        $filter['id'] === 'search_api_fulltext') {
        $filter_key = $filter['expose']['identifier'];
        break;
      }
    }
    if ($query !== '') {
      $view->setExposedInput([$filter_key => $query]);
    }

    // Pager element-aware Request.
    $pager_plugin = $view->display_handler->getPlugin('pager');
    $element = 0;
    if ($pager_plugin && method_exists($pager_plugin, 'getPagerId')) {
      $element = (int) $pager_plugin->getPagerId();
    }
    elseif ($view->getPager() &&
      method_exists($view->getPager(), 'getPagerId')) {
      $element = (int) $view->getPager()->getPagerId();
    }

    $current = $this->requestStack->getCurrentRequest();
    $sub = $current->duplicate();
    if ($query !== '') {
      $sub->query->set($filter_key, $query);
    }
    // Views expects page[<element>]=N.
    $sub->query->set('page', [$element => $page]);

    $view->setRequest($sub);

    // Execute with page set BEFORE execution.
    $view->preExecute();
    if ($view->getPager()) {
      // Zero-based.
      $view->getPager()->setCurrentPage($page);
    }
    elseif (method_exists($view, 'setCurrentPage')) {
      $view->setCurrentPage($page);
    }
    $view->executeDisplay($display_id);

    // Render.
    $build = $view->render();
    if (is_array($build)) {
      $build['#cache']['max-age'] = 0;
    }
    $html = $this->renderer->renderRoot($build);

    return new JsonResponse(['html' => $html]);
  }

}
