<?php

declare(strict_types=1);

namespace Drupal\ai_search_block\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides AI response + native Views results.
 *
 * @Block(
 *   id = "ai_search_block_response",
 *   admin_label = @Translation("AI Search Response"),
 *   category = @Translation("AI Search Block")
 * )
 */
class SearchResponseBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The block plugin manager.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  protected $blockManager;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * Constructs a new SearchResponseBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param mixed $entityTypeManager
   *   The entity type manager.
   * @param mixed $blockManager
   *   The block plugin manager.
   * @param mixed $themeManager
   *   The theme manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, mixed $entityTypeManager, mixed $blockManager, mixed $themeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entityTypeManager;
    $this->blockManager = $blockManager;
    $this->themeManager = $themeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.block'),
      $container->get('theme.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      // Remove the database settings from here.
      // They should come from the form block.
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];
    $build['response_wrapper'] = [
      '#theme' => 'ai_search_block_response',
      // Empty space to ensure div renders.
      '#output' => ' ',
      // Add this if your template expects it.
      '#rendered_form' => NULL,
      // Add this if your template expects it.
      '#suffix_text' => '',
      '#weight' => 0,
    ];

    // Use the injected entity type manager.
    $blocks = $this->entityTypeManager->getStorage('block')->loadByProperties([
      'plugin' => 'ai_search_block',
      'theme' => $this->getTheme(),
    ]);

    /** @var \Drupal\block\BlockInterface $form_block */
    $form_block = reset($blocks);
    // Get the first matching form block.
    if ($form_block) {
      $settings = $form_block->get('settings');

      // Use the form block's database settings.
      if (!empty($settings['enable_database_results']) && !empty($settings['database_results_view'])) {
        [$view_id, $display_id] = explode(':', $settings['database_results_view']);
        $plugin_id = "views_block:{$view_id}-{$display_id}";

        $plugin_block = $this->blockManager->createInstance($plugin_id, []);
        $views_build = $plugin_block->build();
        $views_build['#attributes']['id'] = 'ai-db-results-block';
        $build['db_results'] = $views_build + ['#weight' => 100];
      }
    }

    return $build;
  }

  /**
   * Gets the current theme name.
   *
   * @return string
   *   The current theme name.
   */
  protected function getTheme(): string {
    return $this->themeManager->getActiveTheme()->getName();
  }

}
