<?php

namespace Drupal\ai_seolink_advisor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\ai_seolink_advisor\Analyzer;
use Drupal\ai_seolink_advisor\Service\MyApiService;
use Drupal\ai_seolink_advisor\Service\OpenAIService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns response for seo.
 */
class SeolinkcheckController extends ControllerBase {

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The API service.
   *
   * @var \Drupal\ai_seolink_advisor\Service\MyApiService
   */
  protected $myApiService;

  /**
   * Background classes for status colors.
   *
   * @var array
   */
  protected $backgroundClasses;

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_seolink_advisor\Service\OpenAIService
   */
  protected $openAIService;

  /**
   * {@inheritdoc}
   */
  public function __construct(Request $request, OpenAIService $openAIService, MyApiService $myApiService) {
    $this->request = $request;
    $this->openAIService = $openAIService;
    $this->myApiService = $myApiService;
    $this->backgroundClasses = [
      0 => 'bg-success',
      1 => 'bg-warning',
      2 => 'bg-warning',
      3 => 'bg-warning',
      4 => 'bg-warning',
      5 => 'bg-warning',
      6 => 'bg-danger',
      7 => 'bg-danger',
      8 => 'bg-danger',
      9 => 'bg-danger',
      10 => 'bg-danger',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('ai_seolink_advisor.openai_service'),
      $container->get('ai_seolink_advisor.client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function generateAnalyzerPage() {
    $url = $this->request->query->get('url');
    $config = \Drupal::config('ai_seolink_advisor.settings');
    $ai_service = $config->get('ai_service');

    $output = [
      '#theme' => 'page_ai_seolink_advisor',
      '#search_form' => \Drupal::formBuilder()->getForm('Drupal\ai_seolink_advisor\Form\KeywordForm'),
      '#attached' => ['library' => ['ai_seolink_advisor/styling']],
    ];

    if (empty($url)) {
      return $output;
    }

    $results = [];
    $recommendations = [];
    try {
      $results = (new Analyzer())->analyzeUrl($url, '', $this->languageManager()->getCurrentLanguage()->getId());
      $text = 'Give me SEO recommendations for the URL: ' . $url;

      if ($ai_service != 'any') {
        if ($ai_service === 'openai') {
          $recommendations = $this->openAIService->getSeoRecommendations($url);
        }
        elseif ($ai_service === 'gemini') {
          $recommendations = $this->myApiService->generateContent($text);
        }
        $output['#content_analytics'] = $this->createContentAnalyticsForm($results, $url, $url, $ai_service);
        $output['#general_analytics'] = $this->createGeneralAnalyticsForm($results, $url, $url);
        $output['#recommendations'] = $recommendations;
      }
      else {
        // $config_url = \Drupal::URL('ai_seolink_advisor.settings');
        $config_url = Url::fromRoute('ai_seolink_advisor.settings')->toString();
        $output['#error_message'] = $this->t('Kindly enter the API key after selecting an AI service. <a href="@config_url">Click here</a> to configure the AI service.', ['@config_url' => $config_url]);
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_seolink_advisor')->error('Error analyzing URL: @url. Error: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);

      $output['#error_message'] = $this->t('The URL could not be analyzed. Please verify that the URL is correct and try again.');
    }
    return $output;
  }

  /**
   * {@inheritdoc}
   */
  private function createContentAnalyticsForm($results, $url, $page_url, $ai_service) {
    $content_analytics_form = [];
    $content_analytics_form['content_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("Content analytics for @url", ['@url' => $url]),
      '#open' => TRUE,
    ];
    $data = [
      'meta_tag_length' => $results['PageMeta'],
      'heading_structure' => $results['PageHeaders'],
      'page_content_ratio' => $results['PageContentRatio'],
      'image_alt_texts' => $results['PageAlts'],
      'page_url_size' => $results['PageUrlLength'],
    ];
    if ($ai_service === 'gemini') {
      $data['ai_services'] = 'gemini';
      $data['meta_tag_length_ai'] = $this->myApiService->generateContent($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

      $data['heading_structure_ai'] = $this->myApiService->generateContent($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

      $data['page_content_ratio_ai'] = $this->myApiService->generateContent($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

      $data['image_alt_texts_ai'] = $this->myApiService->generateContent($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

      $data['page_url_size_ai'] = $this->myApiService->generateContent($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
    }
    elseif ($ai_service === 'openai') {
      $data['ai_services'] = 'openai';
      $data['meta_tag_length_ai'] = $this->openAIService->getSeoRecommendations($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

      $data['heading_structure_ai'] = $this->openAIService->getSeoRecommendations($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

      $data['page_content_ratio_ai'] = $this->openAIService->getSeoRecommendations($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

      $data['image_alt_texts_ai'] = $this->openAIService->getSeoRecommendations($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

      $data['page_url_size_ai'] = $this->openAIService->getSeoRecommendations($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
    }

    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('ai_seolink_advisor');
    
    // Generate the base path to the module's images directory.
    $base_path = base_path() . $module_path . '/icons/';
    $content_analytics_form['content_analytics']['content'] = [
      '#theme' => 'table_content_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => $data,
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $content_analytics_form;
  }

  /**
   * {@inheritdoc}
   */
  private function createGeneralAnalyticsForm($results, $url, $page_url) {
    $general_analytics_form = [];
    $general_analytics_form['general_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("General analytics for @url", ['@url' => $url]),
      '#open' => TRUE,
    ];
    
    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('ai_seolink_advisor');
    
    // Generate the base path to the module's images directory.
    $base_path = base_path() . $module_path . '/icons/';

    $general_analytics_form['general_analytics']['content'] = [
      '#theme' => 'table_general_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'is_encrypted' => $results['PageSSL'],
        'has_redirect' => $results['PageRedirect'],
        'content_size' => $results['PageContentSize'],
        'page_load_time' => $results['PageLoadTime'],
        'robots_txt' => $results['FileRobots'],
        'sitemap' => $results['FileSitemap'],
      ],
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $general_analytics_form;
  }

  /**
   * {@inheritdoc}
   */
  private function createRecommendationsForm($recommendations) {
    $form = [];
    $form['recommendations'] = [
      '#type' => 'details',
      '#title' => $this->t("SEO Recommendations"),
      '#open' => TRUE,
    ];
    $form['recommendations']['content'] = [
      '#theme' => 'page_ai_seolink_advisor',
      '#data' => $recommendations,
    ];
    return $form;
  }

}
