<?php

namespace Drupal\ai_seolink_advisor\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;

/**
 * Gemini service.
 */
class MyApiService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The key repository interface.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * Constructs a MyApiService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The Key Repository interface.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, KeyRepositoryInterface $key_repository) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->keyRepository = $key_repository;
  }

  /**
   * {@inheritdoc}
   */
  protected function getApiKey() {
    $config = $this->configFactory->get('ai_seolink_advisor.settings');
    $key_name = $config->get('gemini_api_key');

    if ($key_name) {
      $key = $this->keyRepository->getKey($key_name);
      if ($key) {
        return $key->getKeyValue();
      }
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  protected function getModel() {
    return $this->configFactory->get('ai_seolink_advisor.settings')->get('gemini_model');
  }

  /**
   * {@inheritdoc}
   */
  public function generateContent($text) {
    $gemini_api_key = $this->getApiKey();
    $gemini_model = $this->getModel();
    $version = '';
    if ($gemini_model == 'gemini-1.0-pro') {
      $version = 'v1';
    }
    elseif ($gemini_model == 'gemini-1.5-flash-latest') {
      $version = 'v1beta';
    }

    $url = "https://generativelanguage.googleapis.com/$version/models/$gemini_model:generateContent?key=" . $gemini_api_key;
    $body = [
      'contents' => [
        [
          'parts' => [
            ['text' => $text],
          ],
        ],
      ],
    ];

    try {
      $response = $this->httpClient->post($url, [
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'json' => $body,
      ]);

      $data = json_decode($response->getBody(), TRUE);
      $suggestions = [];

      if (isset($data['candidates']) && is_array($data['candidates'])) {
        foreach ($data['candidates'] as $candidate) {
          if (isset($candidate['content']['parts'][0]['text'])) {
            $suggestions[] = $this->formatMarkdownToHtml($candidate['content']['parts'][0]['text']);
          }
        }
      }
      return $suggestions;
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_seolink_advisor')->error($e->getMessage());
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function formatMarkdownToHtml($text) {
    // Convert headings.
    $text = preg_replace('/\*\*H1:\*\* (.+)/i', '<h1>$1</h1>', $text);
    $text = preg_replace('/\*\*H2:\*\* (.+)/i', '<h2>$1</h2>', $text);
    $text = preg_replace('/\*\*H3:\*\* (.+)/i', '<h3>$1</h3>', $text);
    $text = preg_replace('/\*\*H4:\*\* (.+)/i', '<h4>$1</h4>', $text);

    // Convert bold text.
    $text = preg_replace('/\*\*(.+?)\*\*/', '<b>$1</b>', $text);

    // Add <br> after periods.
    $text = preg_replace('/\.(\s|$)/', '.<br>$1', $text);

    // Add <p> tags around each list item.
    $text = preg_replace('/\*\s(.+)/', '<p>$1</p>', $text);

    return $text;
  }

}
