<?php

namespace Drupal\ai_seolink_advisor\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * OpenAI service.
 */
class OpenAIService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  private $client;

  /**
   * The API key.
   *
   * @var string
   */
  private $apiKey;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private $logger;

  /**
   * The key repository interface.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * Constructs an OpenAIService object.
   *
   * @param \GuzzleHttp\Client $client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The Key Repository interface.
   */
  public function __construct(Client $client, ConfigFactoryInterface $configFactory, LoggerInterface $logger, KeyRepositoryInterface $key_repository) {
    $this->client = new Client(['base_uri' => 'https://api.openai.com/v1/']);
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->keyRepository = $key_repository;
  }

  /**
   * {@inheritdoc}
   */
  public function getApiKey() {
    $config = $this->configFactory->get('ai_seolink_advisor.settings');
    $key_name = $config->get('openai_api_key');

    if ($key_name) {
      $key_name = $this->keyRepository->getKey($key_name);
      if ($key_name) {
        return $key_name->getKeyValue();
      }
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getSeoRecommendations($content) {
    $openai_api_key = $this->getApiKey();
    $payload = [
      'model' => 'gpt-3.5-turbo',
      'messages' => [
        [
          'role' => 'system',
          'content' => 'You are an SEO expert. Provide SEO recommendations for the given content.',
        ],
        [
          'role' => 'user',
          'content' => $content,
        ],
      ],
    ];
    try {
      $response = $this->client->post('chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $openai_api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
      ]);
      $data = json_decode($response->getBody()->getContents(), TRUE);
      return $data['choices'][0]['message']['content'];
    }
    catch (GuzzleException $e) {
      watchdog_exception('ai_seolink_advisor', $e);
      return [
        'error' => TRUE,
        'message' => 'Failed to get SEO recommendations from OpenAI.',
        'details' => $e->getMessage(),
      ];
    }
  }

}
