<?php

namespace Drupal\ai_seo_link_advisor\Controller;

use Drupal\ai_seo_link_advisor\Analyzer;
use Drupal\ai_seo_link_advisor\Service\DynamicAiServiceLoader;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns response for SEO.
 */
class SeolinkcheckController extends ControllerBase {

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The Dynamic AI Service Loader.
   *
   * @var \Drupal\ai_seo_link_advisor\Service\DynamicAiServiceLoader
   */
  protected $dynamicAiServiceLoader;

  /**
   * Background classes for status colors.
   *
   * @var array
   */
  protected $backgroundClasses;

  /**
   * {@inheritdoc}
   */
  public function __construct(Request $request, DynamicAiServiceLoader $dynamicAiServiceLoader) {
    $this->request = $request;
    $this->dynamicAiServiceLoader = $dynamicAiServiceLoader;
    $this->backgroundClasses = [
      0 => 'bg-success',
      1 => 'bg-warning',
      2 => 'bg-warning',
      3 => 'bg-warning',
      4 => 'bg-warning',
      5 => 'bg-warning',
      6 => 'bg-danger',
      7 => 'bg-danger',
      8 => 'bg-danger',
      9 => 'bg-danger',
      10 => 'bg-danger',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('ai_seo_link_advisor.dynamic_ai_service_loader')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function generateAnalyzerPage() {
    // Retrieve the 'url' query parameter from the request.
    $url = $this->request->query->get('url');
    // Get the configuration for the AI SEO Link Advisor module.
    $config = \Drupal::config('ai_seo_link_advisor.settings');
    $ai_service = $config->get('ai_service');

    // Initialize the output array for rendering the page.
    $output = [
      '#theme' => 'page_ai_seo_link_advisor',
      // Add the Seourl form to the page.
      '#search_form' => \Drupal::formBuilder()->getForm('Drupal\ai_seo_link_advisor\Form\SeourlForm'),
      // Attach the required library for styling.
      '#attached' => ['library' => ['ai_seo_link_advisor/styling']],
    ];

    // If the URL is not provided, return the basic output with the search form.
    if (empty($url)) {
      return $output;
    }

    // Initialize arrays to hold analysis results and SEO recommendations.
    $results = [];
    $recommendations = [];
    try {
      // Analyze the URL and get SEO results.
      $results = (new Analyzer())->analyzeUrl($url, '', $this->languageManager()->getCurrentLanguage()->getId());
      $text = 'Give me SEO recommendations for the URL: ' . $url;
      // Get the AI service dynamically.
      $ai_service_instance = $this->dynamicAiServiceLoader->getAiService();
      if ($ai_service_instance) {
        if ($ai_service === 'openai') {
          $recommendations = $ai_service_instance->getSeoRecommendations($url);
        }
        elseif ($ai_service === 'gemini') {
          $recommendations = $ai_service_instance->generateContent($text);
        }
        // Add detailed content analytics and general analytics to the output.
        $output['#content_analytics'] = $this->createContentAnalyticsForm($results, $url, $url, $ai_service);
        $output['#general_analytics'] = $this->createGeneralAnalyticsForm($results, $url, $url);
        $output['#recommendations'] = $recommendations;
      }
      else {
        // Generate a link to setting page if the AI service is not configured.
        $config_url = Url::fromRoute('ai_seo_link_advisor.settings')->toString();
        $output['#error_message'] = $this->t('Kindly enter the API key after selecting an AI service. <a href="@config_url">Click here</a> to configure the AI service.', ['@config_url' => $config_url]);
      }
    }
    catch (\Exception $e) {
      // Log any errors encountered during URL analysis.
      \Drupal::logger('ai_seo_link_advisor')->error('Error analyzing URL: @url. Error: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);

      // Provide a user-friendly error message if analysis fails.
      $output['#error_message'] = $this->t('The URL could not be analyzed. Please verify that the URL is correct and try again.');
    }
    // Return the constructed output array for rendering.
    return $output;
  }

  /**
   * Creates a form for displaying content analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   * @param string $ai_service
   *   The AI service used for recommendations.
   *
   * @return array
   *   A renderable array for content analytics.
   */
  private function createContentAnalyticsForm($results, $url, $page_url, $ai_service) {
    // Initialize the content analytics form array.
    $content_analytics_form = [];
    $content_analytics_form['content_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("Content analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];
    // Prepare data for the content analytics table.
    $data = [
      'meta_tag_length' => $results['PageMeta'],
      'heading_structure' => $results['PageHeaders'],
      'page_content_ratio' => $results['PageContentRatio'],
      'image_alt_texts' => $results['PageAlts'],
      'page_url_size' => $results['PageUrlLength'],
    ];
    // Add AI-specific data if using Gemini or OpenAI services.
  if ($ai_service === 'gemini') {
        $data['ai_services'] = 'gemini';
        $data['meta_tag_length_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

        $data['heading_structure_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

        $data['page_content_ratio_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

        $data['image_alt_texts_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

        $data['page_url_size_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
      }
    elseif ($ai_service === 'openai') {
      $data['ai_services'] = 'openai';
      $data['meta_tag_length_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

      $data['heading_structure_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

      $data['page_content_ratio_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

      $data['image_alt_texts_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

      $data['page_url_size_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
    }
    // Get the module path for images.
    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';
    // Create the content analytics form with the data and icons.
    $content_analytics_form['content_analytics']['content'] = [
      '#theme' => 'table_content_analytics',
      '#page_url' => $page_url,
      '#data' => $data,
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $content_analytics_form;
  }

  /**
   * Creates a form for displaying general analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   *
   * @return array
   *   A renderable array for general analytics.
   */
  private function createGeneralAnalyticsForm($results, $url, $page_url) {
    // Initialize the general analytics form array.
    $general_analytics_form = [];
    $general_analytics_form['general_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("General analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];

    // Get the module path for images.
    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';

    // Create the general analytics form with the data and icons.
    $general_analytics_form['general_analytics']['content'] = [
      '#theme' => 'table_general_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'is_encrypted' => $results['PageSSL'],
        'has_redirect' => $results['PageRedirect'],
        'content_size' => $results['PageContentSize'],
        'page_load_time' => $results['PageLoadTime'],
        'robots_txt' => $results['FileRobots'],
        'sitemap' => $results['FileSitemap'],
      ],
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $general_analytics_form;
  }

  /**
   * Creates a form to display SEO recommendations.
   *
   * This method constructs a form array that includes SEO recommendations.
   *
   * @param array $recommendations
   *   An associative array of SEO recommendations to be displayed in the form.
   *
   * @return array
   *   A renderable array representing the form structure.
   */
  private function createRecommendationsForm($recommendations) {
    // Initialize the form array.
    $form = [];
    // Define a 'details' element to wrap the SEO recommendations.
    $form['recommendations'] = [
      '#type' => 'details',
      '#title' => $this->t("SEO Recommendations"),
      '#open' => TRUE,
    ];
    // Add the actual content of the recommendations inside the details element.
    $form['recommendations']['content'] = [
      '#theme' => 'page_ai_seo_link_advisor',
      '#data' => $recommendations,
    ];
    return $form;
  }

}
