<?php

namespace Drupal\ai_seo_link_advisor\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure AI service settings.
 */
class AiapiConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_api_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'ai_seo_link_advisor.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Load the current configuration settings.
    $config = $this->config('ai_seo_link_advisor.settings');
    // Define the AI service selection field.
    $form['ai_service'] = [
      '#type' => 'select',
      '#title' => $this->t('Select AI Service'),
      '#options' => [
        'any' => $this->t('Select AI Service'),
        'openai' => $this->t('OpenAI'),
        'gemini' => $this->t('Gemini'),
      ],
      '#default_value' => $config->get('ai_service') ?: 'any',
      '#description' => $this->t('Select the AI service to use.'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateForm',
        'wrapper' => 'ai-service-config',
      ],
    ];
    // Define a container for the AI service-specific configuration fields.
    $form['ai_service_config'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ai-service-config'],
    ];

    // Retrieve the selected AI service from the form state or configuration.
    $selected_ai_service = $form_state->getValue('ai_service', $config->get('ai_service'));
    // Define fields for OpenAI service configuration if selected.
    if ($selected_ai_service === 'openai') {
      $form['ai_service_config']['openai_api_key'] = [
        '#type' => 'key_select',
        '#title' => $this->t('OpenAI API Key'),
        '#default_value' => $config->get('openai_api_key'),
        '#description' => $this->t('Enter the OpenAI API key.'),
        '#required' => FALSE,
      ];

      $form['ai_service_config']['openai_model'] = [
        '#type' => 'select',
        '#title' => $this->t('OpenAI Model'),
        '#options' => [
          'gpt-3.5-turbo' => $this->t('OpenAI GPT-3.5 Turbo'),
          'gpt-4' => $this->t('OpenAI GPT-4'),
        ],
        '#default_value' => $config->get('openai_model'),
        '#description' => $this->t('Select the OpenAI model to use.'),
        '#required' => TRUE,
      ];
    }
    // Define fields for Gemini service configuration if selected.
    elseif ($selected_ai_service === 'gemini') {
      $form['ai_service_config']['gemini_api_key'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Gemini API Key'),
        '#default_value' => $config->get('gemini_api_key'),
        '#description' => $this->t('Enter the Gemini API key.'),
        '#required' => FALSE,
      ];

      $form['ai_service_config']['gemini_model'] = [
        '#type' => 'select',
        '#title' => $this->t('Gemini Model'),
        '#options' => [
          'gemini-1.0-pro' => $this->t('Gemini 1.0 Pro'),
          'gemini-1.5-flash-latest' => $this->t('Gemini 1.5 Flash Latest'),
          // Add other models here.
        ],
        '#default_value' => $config->get('gemini_model'),
        '#description' => $this->t('Select the Gemini model to use.'),
        '#required' => TRUE,
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Call parent validation method to maintain default validation behavior.
    parent::validateForm($form, $form_state);

    // Validate the presence of API keys based on the selected service.
    $selected_ai_service = $form_state->getValue('ai_service');
    if ($selected_ai_service === 'openai' && empty($form_state->getValue('openai_api_key'))) {
      $form_state->setErrorByName('openai_api_key', $this->t('The OpenAI API key is required.'));
    }
    elseif ($selected_ai_service === 'gemini' && empty($form_state->getValue('gemini_api_key'))) {
      $form_state->setErrorByName('gemini_api_key', $this->t('The Gemini API key is required.'));
    }
  }

  /**
   * Ajax callback to update the form based on the selected AI service.
   *
   * This method updates the form elements dynamically based on user selection.
   */
  public function updateForm(array &$form, FormStateInterface $form_state) {
    return $form['ai_service_config'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save the configuration settings based on form input.
    $this->config('ai_seo_link_advisor.settings')
      ->set('ai_service', $form_state->getValue('ai_service'))
      ->set('openai_api_key', $form_state->getValue('openai_api_key'))
      ->set('openai_model', $form_state->getValue('openai_model'))
      ->set('gemini_api_key', $form_state->getValue('gemini_api_key'))
      ->set('gemini_model', $form_state->getValue('gemini_model'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
