<?php

namespace Drupal\ai_seo_link_advisor\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;

/**
 * Service class for interacting with the Gemini API.
 */
class GeminiApiService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The key repository interface.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * Constructs a GeminiApiService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The Key Repository interface.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, KeyRepositoryInterface $key_repository) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->keyRepository = $key_repository;
  }

  /**
   * {@inheritdoc}
   */
  protected function getApiKey() {
    // Load the module configuration.
    $config = $this->configFactory->get('ai_seo_link_advisor.settings');
    // Retrieve the API key name from the configuration.
    $key_name = $config->get('gemini_api_key');
    // If the key name is set, fetch the actual key value.
    if ($key_name) {
      $key = $this->keyRepository->getKey($key_name);
      if ($key) {
        return $key->getKeyValue();
      }
    }

    return '';
  }

  /**
   * Retrieves the selected model from the configuration settings.
   *
   * @return string
   *   The model name.
   */
  protected function getModel() {
    return $this->configFactory->get('ai_seo_link_advisor.settings')->get('gemini_model');
  }

  /**
   * Generates content using the Gemini API.
   *
   * @param string $text
   *   The input text to be processed.
   *
   * @return array|null
   *   An array of formatted suggestions or NULL on failure.
   */
  public function generateContent($text) {
    // Get the API key and model from the configuration.
    $gemini_api_key = $this->getApiKey();
    $gemini_model = $this->getModel();
    $version = '';
    // Determine the API version based on the selected model.
    if ($gemini_model == 'gemini-1.0-pro') {
      $version = 'v1';
    }
    elseif ($gemini_model == 'gemini-1.5-flash-latest') {
      $version = 'v1beta';
    }
    // Construct the API endpoint URL.
    $url = "https://generativelanguage.googleapis.com/$version/models/$gemini_model:generateContent?key=" . $gemini_api_key;
    // Prepare the request body.
    $body = [
      'contents' => [
        [
          'parts' => [
            ['text' => $text],
          ],
        ],
      ],
    ];

    try {
      // Send the request to the Gemini API.
      $response = $this->httpClient->post($url, [
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'json' => $body,
      ]);
      // Parse the JSON response.
      $data = json_decode($response->getBody(), TRUE);
      $suggestions = [];
      // Extract and format the suggestions from the response.
      if (isset($data['candidates']) && is_array($data['candidates'])) {
        foreach ($data['candidates'] as $candidate) {
          if (isset($candidate['content']['parts'][0]['text'])) {
            $suggestions[] = $this->formatMarkdownToHtml($candidate['content']['parts'][0]['text']);
          }
        }
      }
      return $suggestions;
    }
    catch (\Exception $e) {
      // Log any errors encountered during the request.
      \Drupal::logger('ai_seo_link_advisor')->error($e->getMessage());
      return NULL;
    }
  }

  /**
   * Converts markdown-like syntax to HTML.
   *
   * @param string $text
   *   The text to be formatted.
   *
   * @return string
   *   The formatted HTML string.
   */
  public function formatMarkdownToHtml($text) {
    // Convert markdown headings to HTML.
    $text = preg_replace('/\*\*H1:\*\* (.+)/i', '<h1>$1</h1>', $text);
    $text = preg_replace('/\*\*H2:\*\* (.+)/i', '<h2>$1</h2>', $text);
    $text = preg_replace('/\*\*H3:\*\* (.+)/i', '<h3>$1</h3>', $text);
    $text = preg_replace('/\*\*H4:\*\* (.+)/i', '<h4>$1</h4>', $text);

    // Convert markdown bold text to HTML.
    $text = preg_replace('/\*\*(.+?)\*\*/', '<b>$1</b>', $text);

    // Add line breaks after periods.
    $text = preg_replace('/\.(\s|$)/', '.<br>$1', $text);

    // Wrap each list item in <p> tags.
    $text = preg_replace('/\*\s(.+)/', '<p>$1</p>', $text);

    return $text;
  }

}
