<?php

namespace Drupal\ai_seo_link_advisor;

use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\Client;
use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface;
use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\Exception\HttpException;
use Drupal\ai_seo_link_advisor\Analyzer\Metric\KeywordBasedMetricInterface;
use Drupal\ai_seo_link_advisor\Analyzer\Metric\MetricFactory;
use Drupal\ai_seo_link_advisor\Analyzer\Parser\Parser;
use Drupal\ai_seo_link_advisor\Analyzer\Parser\ParserInterface;

/**
 * Provides a Page class for SEO analysis.
 */
class Page {
  /**
   * Language code constant.
   */
  const LANGCODE = 'langcode';

  /**
   * Stop words constant.
   */
  const STOP_WORDS = 'stop_words';

  /**
   * Keyword constant.
   */
  const KEYWORD = 'keyword';

  /**
   * Impact constant.
   */
  const IMPACT = 'impact';

  /**
   * Text constant.
   */
  const TEXT = 'text';

  /**
   * Headers constant.
   */
  const HEADERS = 'headers';

  /**
   * URL of the web page.
   *
   * @var string
   */
  public $url;

  /**
   * Configuration settings.
   *
   * @var array
   */
  public $config = [];

  /**
   * Language code of the page.
   *
   * @var string
   */
  public $langcode = 'en';

  /**
   * Keyword to use in analysis.
   *
   * @var string
   */
  public $keyword;

  /**
   * Stop words used in keyword density analysis.
   *
   * @var array
   */
  public $stopWords = [];

  /**
   * HTML content of the web page.
   *
   * @var string
   */
  public $content;

  /**
   * Factors values of the web page.
   *
   * @var array
   */
  public $factors = [];

  /**
   * HTTP client interface.
   *
   * @var \Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface
   */
  public $client;

  /**
   * Parser interface.
   *
   * @var \Drupal\ai_seo_link_advisor\Analyzer\Parser\ParserInterface
   */
  public $parser;

  /**
   * Page constructor.
   *
   * @param string|null $url
   *   The URL of the web page.
   * @param string $langcode
   *   The language code of the page.
   * @param \Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface|null $client
   *   The HTTP client interface.
   * @param \Drupal\ai_seo_link_advisor\Analyzer\Parser\ParserInterface|null $parser
   *   The parser interface.
   */
  public function __construct(
    string $url = NULL,
    $langcode = 'en',
    ClientInterface $client = NULL,
    ParserInterface $parser = NULL,
  ) {
    $this->client = $client ?? new Client();
    $this->parser = $parser ?? new Parser();

    if (!empty($url)) {
      $this->url = $this->setUpUrl($url);
      $this->getContent();
    }

    $this->config = [
      'langcode' => 'en',
      'client' => new Client(),
      'parser' => new Parser(),
      'factors' => [
        Factor::SSL,
        Factor::REDIRECT,
        Factor::CONTENT_SIZE,
        Factor::META,
        Factor::HEADERS,
        Factor::CONTENT_RATIO,
        [Factor::DENSITY_PAGE => 'keywordDensity'],
        [Factor::DENSITY_HEADERS => 'headersKeywordDensity'],
        Factor::ALTS,
        Factor::URL_LENGTH,
        Factor::LOAD_TIME,
        [Factor::KEYWORD_URL => 'keywordUrl'],
        [Factor::KEYWORD_PATH => 'keywordPath'],
        [Factor::KEYWORD_TITLE => 'keywordTitle'],
        [Factor::KEYWORD_DESCRIPTION => 'keywordDescription'],
        Factor::KEYWORD_HEADERS,
        [Factor::KEYWORD_DENSITY => 'keywordDensity'],
      ],
    ];
    $this->config['langcode'] = $langcode;
    $this->langcode = $langcode;
  }

  /**
   * Verifies URL and sets up some basic metrics.
   *
   * @param string $url
   *   The URL to set up.
   *
   * @return string
   *   The verified URL.
   */
  protected function setUpUrl(string $url): string {
    $parsedUrl = parse_url($url);
    if (empty($parsedUrl['scheme'])) {
      $url = 'http://' . $url;
      $parsedUrl = parse_url($url);
    }
    $this->setFactor(Factor::URL_PARSED, $parsedUrl);

    if (strcmp($parsedUrl['scheme'], 'https') === 0) {
      $this->setFactor(Factor::SSL, TRUE);
    }
    $this->setFactor(
      Factor::URL_LENGTH,
      strlen($this->getFactor(Factor::URL_PARSED_HOST) . $this->getFactor(Factor::URL_PARSED_PATH))
    );
    return $url;
  }

  /**
   * Downloads page content from URL specified and sets up some base metrics.
   */
  public function getContent() {
    $pageLoadFactors = $this->getPageLoadFactors();
    $this->setFactor(Factor::LOAD_TIME, $pageLoadFactors['time']);
    $this->content = $pageLoadFactors['content'];
    $this->setFactor(Factor::REDIRECT, $pageLoadFactors['redirect']);
    if (empty($this->getFactor(Factor::SSL)) && $this->getSslResponseCode() == 200) {
      $this->setFactor(Factor::SSL, TRUE);
    }
  }

  /**
   * Sets page load related factors.
   *
   * @return array
   *   The verified URL.
   */
  protected function getPageLoadFactors(): array {

    $starTime = microtime(TRUE);
    $response = $this->client->get($this->url);
    $loadTime = number_format((microtime(TRUE) - $starTime), 4);
    $redirect = [
      'redirect' => FALSE,
      'source' => $this->url,
      'destination' => FALSE,
    ];
    if (!empty($redirects = $response->getHeader('X-Guzzle-Redirect-History'))) {
      $redirect = [
        'redirect' => TRUE,
        'source' => $this->url,
        'destination' => end($redirects),
      ];
    }
    return [
      'content' => $response->getBody()->getContents(),
      'time' => $loadTime,
      'redirect' => $redirect,
    ];
  }

  /**
   * Returns https response code.
   *
   * @return int|false
   *   Http code or false on failure.
   */
  protected function getSslResponseCode() {
    try {
      return $this->client->get(str_replace('http://', 'https://', $this->url))->getStatusCode();
    }
    catch (HttpException $e) {
      return FALSE;
    }
  }

  /**
   * Parses page's html content setting up related metrics.
   */
  public function parse() {
    if (empty($this->content)) {
      $this->getContent();
    }

    $this->parser->setContent($this->content);
    $this->setFactors([
      Factor::META_META => $this->parser->getMeta(),
      Factor::HEADERS => $this->parser->getHeaders($this->keyword),
      Factor::META_TITLE => $this->parser->getTitle(),
      Factor::TEXT => $this->parser->getText(),
      Factor::ALTS => $this->parser->getImages(),
    ]);
  }

  /**
   * Returns page metrics.
   *
   * @return array
   *   The verified URL.
   */
  public function getMetrics(): array {
    $this->initializeFactors();
    $metrics = $this->setUpMetrics($this->config['factors']);
    return $metrics;
  }

  /**
   * Sets up and returns page metrics based on configuration specified.
   *
   * @param array $config
   *   The URL to set up.
   *
   * @return array
   *   The verified URL.
   */
  public function setMetrics(array $config) {
    $this->initializeFactors();
    return $this->setUpMetrics($config);
  }

  /**
   * Sets up page content related factors for page metrics.
   */
  private function initializeFactors() {
    if (empty($this->dom)) {
      $this->parse();
    }
    $this->setUpContentFactors();
    if (!empty($this->keyword)) {
      $this->setUpContentKeywordFactors($this->keyword);
    }
  }

  /**
   * Sets up page content related factors for page metrics.
   */
  public function setUpContentFactors() {
    $this->setFactors([
      Factor::CONTENT_HTML => $this->content,
      Factor::CONTENT_SIZE => strlen($this->content),
      Factor::CONTENT_RATIO => [
        'content_size' => strlen(preg_replace('!\s+!', ' ', $this->getFactor(Factor::TEXT))),
        'code_size' => strlen($this->content),
      ],
      Factor::DENSITY_PAGE => [
        self::TEXT => $this->getFactor(Factor::TEXT),
        self::LANGCODE => $this->langcode,
        self::STOP_WORDS => $this->stopWords,
      ],
      Factor::DENSITY_HEADERS => [
        self::HEADERS => $this->getFactor(Factor::HEADERS),
        self::LANGCODE => $this->langcode,
        self::STOP_WORDS => $this->stopWords,
      ],
    ]);
  }

  /**
   * Sets up page content factors keyword related.
   *
   * @param string $keyword
   *   The URL to set up.
   */
  public function setUpContentKeywordFactors(string $keyword) {

    $meta_description = $this->getFactor(Factor::META)['meta']['description'] ?? $this->getFactor(Factor::META_DESCRIPTION);
    $meta_title = $this->getFactor(Factor::META)['title'] ?? $this->getFactor(Factor::META_TITLE);
    if ($keyword && !empty($keyword)) {
      $meta_description = str_ireplace($keyword, "<strong>$keyword</strong>", $meta_description);
      $meta_title = str_ireplace($keyword, "<strong>$keyword</strong>", $meta_title);
    }

    $this->setFactors([
      Factor::KEYWORD_URL => [
        self::TEXT => $this->getFactor(Factor::URL_PARSED_HOST),
        self::KEYWORD => $keyword,
        self::IMPACT => 5,
        'type' => 'URL',
      ],
      Factor::KEYWORD_PATH => [
        self::TEXT => $this->getFactor(Factor::URL_PARSED_PATH),
        self::KEYWORD => $keyword,
        self::IMPACT => 3,
        'type' => 'UrlPath',
      ],
      Factor::KEYWORD_TITLE => [
        self::TEXT => $meta_title,
        self::KEYWORD => $keyword,
        self::IMPACT => 5,
        'type' => 'Title',
      ],
      Factor::KEYWORD_DESCRIPTION => [
        self::TEXT => $meta_description,
        self::KEYWORD => $keyword,
        self::IMPACT => 3,
        'type' => 'Description',
      ],
      Factor::KEYWORD_HEADERS => [
        self::HEADERS => $this->getFactor(Factor::HEADERS),
        self::KEYWORD => $keyword,
      ],
      Factor::KEYWORD_DENSITY => [
        self::TEXT => $this->getFactor(Factor::TEXT),
        self::LANGCODE => $this->langcode,
        self::STOP_WORDS => $this->stopWords,
        self::KEYWORD => $keyword,
      ],
    ]);
  }

  /**
   * Sets up page metrics.
   *
   * @param array $config
   *   The URL to set up.
   *
   * @return array
   *   The verified URL.
   */
  public function setUpMetrics(array $config = []) {
    $metrics = [];
    foreach ($config as $factor) {
      $metric = $factor;
      if (is_array($factor)) {
        $metric = current($factor);
        $factor = key($factor);
      }
      $metricObject = MetricFactory::get('page.' . $metric, $this->getFactor($factor));
      if (!$metricObject instanceof KeywordBasedMetricInterface || !empty($this->keyword)) {
        $metrics['page_' . str_replace('.', '_', $metric)] = $metricObject;
      }
    }
    return $metrics;
  }

  /**
   * Sets page factor value.
   */
  public function setFactor(string $name, $value) {
    if (count(explode('.', $name)) > 1) {
      $this->setArrayByDot($this->factors, $name, $value);
    }
    else {
      $this->factors[$name] = $value;
    }
    return $this->factors;
  }

  /**
   * Sets array values using string with dot notation.
   */
  protected function setArrayByDot(array &$array, string $path, $val) {
    $loc = &$array;
    foreach (explode('.', $path) as $step) {
      $loc = &$loc[$step];
    }
    return $loc = $val;
  }

  /**
   * Sets multiple page factors values at once.
   *
   * @param array $factors
   *   The URL to set up.
   */
  public function setFactors(array $factors) {
    foreach ($factors as $factorName => $factorValue) {
      $this->setFactor($factorName, $factorValue);
    }
  }

  /**
   * Returns factor data collected by it's key name.
   */
  public function getFactor($name) {
    if (strpos($name, '.') !== FALSE) {
      return $this->getNestedFactor($name);
    }
    if (!empty($this->factors[$name])) {
      return $this->factors[$name];
    }
    return FALSE;
  }

  /**
   * Returns factor data collected by it's key name.
   *
   * @param string $name
   *   The URL to set up.
   *
   * @return mixed
   *   The verified URL.
   */
  protected function getNestedFactor($name) {
    $keys = explode('.', $name);
    $value = $this->factors;
    foreach ($keys as $innerKey) {
      if (!array_key_exists($innerKey, $value)) {
        return FALSE;
      }
      $value = $value[$innerKey];
    }
    return $value;
  }

}
