<?php

namespace Drupal\ai_translate_lb_asymmetric\Plugin\FieldTextExtractor;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_translate\Attribute\FieldTextExtractor;
use Drupal\ai_translate\FieldTextExtractorInterface;
use Drupal\ai_translate\Plugin\FieldTextExtractor\LbFieldExtractor;

/**
 * A field text extractor plugin for Layout builder asymmetric fields.
 */
#[FieldTextExtractor(
  id: "layout_builder_asymmetric",
  label: new TranslatableMarkup('Layout builder Asymmetric'),
  field_types: [
    'layout_section',
  ]
)]
class LbAsymmetricExtractor extends LbFieldExtractor implements FieldTextExtractorInterface, ContainerFactoryPluginInterface {

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * The block content storage service.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $blockStorage;

  /**
   * Logger channel for this class.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static();
    $instance->config = $container->get('config.factory')->get('ai_translate.settings');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->sectionStorageManager = $container->get('plugin.manager.layout_builder.section_storage');
    $instance->textExtractor = $container->get('ai_translate.text_extractor');
    $instance->uuid = $container->get('uuid');
    $instance->blockStorage = $container->get('entity_type.manager')->getStorage('block_content');
    $instance->logger = $container->get('logger.factory')->get('ai_translate_lb_asymmetric');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function setValue(ContentEntityInterface $entity, string $fieldName, array $textMeta): void {
    $sourceEntity = $entity->getUntranslated();
    $translationLanguage = $entity->language()->getId();
    $originalSections = $sourceEntity->get($fieldName)->getValue();
    $duplicatedSections = [];

    foreach ($originalSections as $sectionData) {
      $section = $sectionData['section'];
      $duplicatedSection = clone $section;

      foreach ($duplicatedSection->getComponents() as $componentId => $component) {
        $blockEntity = NULL;
        $configuration = $component->get('configuration');

        if (isset($configuration['block_revision_id'])) {
          $blockEntity = $this->blockStorage->loadRevision($configuration['block_revision_id']);
        }
        elseif (isset($configuration['block_id'])) {
          $blockEntity = $this->blockStorage->load($configuration['block_id']);
        }

        if ($blockEntity) {
          $clonedBlock = $blockEntity->createDuplicate();
          $clonedBlock->set('uuid', $this->uuid->generate());
          $clonedBlock->set('langcode', $translationLanguage);

          foreach ($textMeta as $uuid => $translations) {
            if ($uuid === $blockEntity->uuid()) {
              foreach ($translations as $field => $fieldData) {
                foreach ($fieldData as $delta => $translationData) {
                  if (isset($translationData['value']) && $clonedBlock->hasField($field)) {
                    $clonedBlock->set($field, $translationData, $delta);
                  }
                }
              }
            }
          }

          try {
            $clonedBlock->save();
          }
          catch (\Exception $e) {
            $this->logger->error('Failed to save cloned block: @message', ['@message' => $e->getMessage()]);
          }

          // Update the component configuration with the cloned block's IDs.
          $configuration['block_id'] = $clonedBlock->id();
          $configuration['block_revision_id'] = $clonedBlock->getRevisionId();
          $component->setConfiguration($configuration);
          // Generate a new UUID for the component to avoid conflicts.
          $duplicatedSection->getComponent($componentId)->set('uuid', $this->uuid->generate());
        }
      }

      $duplicatedSections[] = $duplicatedSection;
    }

    $entity->get($fieldName)->setValue($duplicatedSections);

    try {
      $entity->save();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to save translated entity: @message', ['@message' => $e->getMessage()]);
    }
  }

}
