<?php

namespace Drupal\ai_translate_paragraph_asymetric\Plugin\FieldTextExtractor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_translate\Attribute\FieldTextExtractor;
use Drupal\ai_translate\ConfigurableFieldTextExtractorInterface;
use Drupal\ai_translate\Plugin\FieldTextExtractor\ReferenceFieldExtractor;
use Drupal\paragraphs\ParagraphInterface;

/**
 * A field text extractor plugin for paragraph asymmetric fields.
 */
#[FieldTextExtractor(
  id: "paragraph_asymmetric",
  label: new TranslatableMarkup('Paragraph Asymmetric'),
  field_types: [
    'entity_reference',
    'entity_reference_revisions',
  ]
)]
class ParagraphAsymetricExtractor extends ReferenceFieldExtractor implements ConfigurableFieldTextExtractorInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function setValue(
    ContentEntityInterface $entity,
    string $fieldName,
    array $textMeta,
  ): void {
    $newValue = [];
    $source_entity = $entity->getUntranslated();
    $referencedEntities = $source_entity->get($fieldName)->referencedEntities();
    $translationLanguage = $entity->language()->getId();

    foreach ($textMeta as $delta => $singleValue) {
      // Original value always exists, otherwise no translation.
      $referencedEntity = $referencedEntities[$delta];
      if ($referencedEntity instanceof ParagraphInterface) {
        // Create a duplicate with the correct translation language.
        $referencedEntity = $this->createDuplicateWithSingleLanguage($referencedEntity, $translationLanguage);

        // Recursively handle any nested paragraphs in the entity's fields.
        foreach ($singleValue as $subFieldName => $subValue) {
          foreach ($subValue as &$singleSubValue) {
            unset($singleSubValue['field_name']);
            unset($singleSubValue['field_type']);
          }

          // Check if the field is referencing other paragraphs.
          if ($referencedEntity->get($subFieldName)->getFieldDefinition()->getType() === 'entity_reference_revisions') {
            $nestedParagraphs = $referencedEntity->get($subFieldName)->referencedEntities();
            if (!empty($nestedParagraphs)) {
              // Recursively process nested paragraphs.
              $this->setNestedParagraphs($referencedEntity, $subFieldName, $subValue, $translationLanguage);
            }
          }
          else {
            // Otherwise, set the simple value.
            $referencedEntity->set($subFieldName, $subValue);
          }
        }

        $newValue[$delta] = ['entity' => $referencedEntity];
      }

      $entity->set($fieldName, $newValue);
    }
  }

  /**
   * Recursively handle nested paragraphs and apply translations.
   */
  protected function setNestedParagraphs(
    ParagraphInterface $parentParagraph,
    string $fieldName,
    array $nestedMeta,
    string $translationLanguage,
  ): void {
    $newNestedValue = [];
    $nestedReferencedEntities = $parentParagraph->get($fieldName)->referencedEntities();

    foreach ($nestedMeta as $delta => $nestedValue) {
      $nestedReferencedEntity = $nestedReferencedEntities[$delta];

      if ($nestedReferencedEntity instanceof ParagraphInterface) {
        // Create duplicate for the nested paragraph with the correct language.
        $nestedReferencedEntity = $this->createDuplicateWithSingleLanguage($nestedReferencedEntity, $translationLanguage);

        // Set the nested values.
        foreach ($nestedValue as $subFieldName => $subValue) {
          foreach ($subValue as &$singleSubValue) {
            unset($singleSubValue['field_name']);
            unset($singleSubValue['field_type']);
          }
          // Apply values to the nested paragraph.
          $nestedReferencedEntity->set($subFieldName, $subValue);
        }

        $newNestedValue[$delta] = ['entity' => $nestedReferencedEntity];
      }
    }

    // Make sure to set html decoded values.
    $parentParagraph->set($fieldName, $newNestedValue);
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(FieldConfigInterface $entity, FormStateInterface $form_state, array &$completeForm = []) {
    return [];
  }

  /**
   * Method to create duplicates of the paragraph.
   */
  protected function createDuplicateWithSingleLanguage(ParagraphInterface $paragraph, $langcode) {
    $duplicate = $paragraph->createDuplicate();

    // Clone all sub-paragraphs recursively.
    foreach ($duplicate->getFields(FALSE) as $field) {
      // @todo should we support field collections as well?
      if ($field->getFieldDefinition()->getType() === 'entity_reference_revisions' && $field->getFieldDefinition()->getTargetEntityTypeId() == 'paragraph') {
        foreach ($field as $item) {
          if (!$item->entity instanceof ParagraphInterface) {
            continue;
          }
          $item->entity = $this->createDuplicateWithSingleLanguage($item->entity, $langcode);
        }
      }
    }

    // Change the original language and remove possible translations.
    if ($duplicate->isTranslatable()) {
      $duplicate->set('langcode', $langcode);
      foreach ($duplicate->getTranslationLanguages(FALSE) as $language) {
        try {
          $duplicate->removeTranslation($language->getId());
        }
        catch (\InvalidArgumentException $e) {
          // Should never happen.
        }
      }
    }

    return $duplicate;
  }

}
