<?php

namespace Drupal\ai_translate_textfield;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Factory class for creating instances of AiTranslatorInterface.
 */
class AiTranslatorFactory {

  /**
   * The service container.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * The configuration of this module.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a AiTranslatorFactory object.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(ContainerInterface $container, ConfigFactoryInterface $configFactory) {
    $this->container = $container;
    $this->config = $configFactory->get('ai_translate_textfield.settings');
  }

  /**
   * Creates an AiTranslatorInterface instance.
   *
   * @param string $serviceName
   *   The name of the service to instantiate.
   *
   * @return \Drupal\ai_translate_textfield\AiTranslatorInterface
   *   An instance of AiTranslatorInterface.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the specified service is not valid.
   */
  public function createTranslationService(string $serviceName) {

    $class = $this->getTranslatorConfig($serviceName, 'class_name');

    // Check if the service class exists.
    if (!class_exists($class)) {
      throw new \InvalidArgumentException('Invalid translation service class.');
    }

    return $class::create($this->container);
  }

  /**
   * Gets the configuration translation service(s).
   *
   * @param ?string $serviceName
   *   The service's machine name. If null, returns all.
   * @param ?string $key
   *   The config key. If null, returns full config array.
   *
   * @return ?array|string
   *   The configuration value: array or string, or NULL if not found.
   */
  public function getTranslatorConfig(?string $serviceName = NULL, ?string $key = NULL): mixed {
    $translatorConfig = $this->config->get('translator_config');

    if (empty($translatorConfig)) {
      return NULL;
    }

    if (is_null($serviceName)) {
      return $translatorConfig;
    }

    if (!isset($translatorConfig[$serviceName])) {
      return NULL;
    }

    if (is_null($key)) {
      return $translatorConfig[$serviceName];
    }

    if (isset($translatorConfig[$serviceName][$key])) {
      return $translatorConfig[$serviceName][$key];
    }

    return NULL;

  }

  /**
   * A helper function to avoid injecting the config in another class.
   *
   * @param string $key
   *   The configuration key.
   *
   * @return mixed
   *   The config data.
   */
  public function getConfig(string $key = ''): mixed {
    return $this->config->get($key);
  }

}
