<?php

namespace Drupal\ai_translate_textfield;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The ChatGPT translator implementation.
 */
class ChatGPTTranslator implements AiTranslatorInterface, ContainerInjectionInterface {

  /**
   * The configuration of this module.
   *
   * @var array
   */
  protected array $config = [];

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

   /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $config = $container->get('config.factory')->get('ai_translate_textfield.settings')->get('translator_config');
    if (!isset($config['chatgpt'])) {
      throw new InvalidConfigurationException('The ChatGPT translator is not configured.');
    }
    $instance->config = $config['chatgpt'];
    $instance->client = $container->get('http_client');
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function translateText(string $text, string $targetLang, bool $isHtml = FALSE) {

    $languageName = $this->languageManager->getLanguageName($targetLang);
    $prompt = "Translate the following text into {$languageName}: {$text}";
    $apiKey = $this->config['api_key'];

    try {
      $response = $this->client->post($this->config['api_endpoint'], [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => "Bearer {$apiKey}",
        ],
        'json' => [
          'messages' => [
            [
              'role' => 'user',
              'content' => $prompt,
            ],
          ],
          'model' => 'gpt-3.5-turbo',
        ],
      ]);
    }
    catch (RequestException $e) {
      return $text;
    }

    $decoded = json_decode($response->getBody());
    $translated = $decoded->choices[0]->message->content;

    return $translated;
  }

}
