<?php

namespace Drupal\ai_translate_textfield\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\Yaml\Exception\ParseException;
use Symfony\Component\Yaml\Yaml;

/**
 * Settings for the ai_translate_textfield module.
 */
class AiTranslateTextfieldSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_translate_textfield_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_translate_textfield.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $config = $this->config('ai_translate_textfield.settings');
    $translatorConfig = $config->get('translator_config');

    $form['options'] = [
      '#title' => $this->t('Generic options'),
      '#type' => 'fieldset',
    ];

    $form['options']['button_text'] = [
      '#type' => 'textfield',
      '#default_value' => $config->get('button_text'),
      '#title' => $this->t('Button text'),
      '#description' => $this->t('The text shown on the translate buttons added to the form elements.'),
    ];

    $form['options']['warning_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Warning modal enabled'),
      '#description' => $this->t('If this is enabled, a warning dialog with following texts is popped up before sending the data to the translator backend.'),
      '#default_value' => $config->get('warning_enabled'),
    ];

    $form['options']['warning_modal'] = [
      '#title' => $this->t('Warning modal dialog'),
      '#type' => 'fieldset',
      '#states' => [
        'visible' => [
          ':input[name="warning_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['options']['warning_modal']['dialog_title'] = [
      '#type' => 'textfield',
      '#default_value' => $config->get('dialog_title'),
      '#title' => $this->t('The title text of the dialog'),
    ];

    $form['options']['warning_modal']['dialog_content'] = [
      '#type' => 'text_format',
      '#default_value' => $config->get('dialog_content'),
      '#title' => $this->t('The text content of the dialog'),
      '#default_value' => $config->get('dialog_content')['value'] ?? '',
      '#format' => $config->get('dialog_content')['format'] ?? filter_default_format(),
    ];

    $form['options']['warning_modal']['dialog_ok_button'] = [
      '#type' => 'textfield',
      '#default_value' => $config->get('dialog_ok_button'),
      '#title' => $this->t('Ok button text'),
      '#description' => $this->t('The text shown on the "Ok" button which makes the text to be sent to the translator.'),
    ];

    $form['options']['warning_modal']['dialog_cancel_button'] = [
      '#type' => 'textfield',
      '#default_value' => $config->get('dialog_cancel_button'),
      '#title' => $this->t('Cancel button text'),
      '#description' => $this->t('The text shown on the "Cancel" button which closes the modal dialog.'),
    ];

    $form['translator_config'] = [
      '#type' => 'fieldset',
    ];

    $form['translator_config']['translator_config'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Translator backend configuration'),
      '#description' => $this->t('Input needed configuration as valid YAML. A "label" and "class_name" keys under the service machine name are required.'),
      '#default_value' => Yaml::dump($translatorConfig),
      // If yaml_editor is installed, this works automatically.
      '#attributes' => ['data-yaml-editor' => 'true'],
    ];
    $form['translator_config']['example'] = [
      '#type' => 'markup',
      '#markup' => "<p>An example of the configuration</p><pre>
deepl:
  label: 'DeepL API'
  class_name: \Drupal\ai_translate_textfield\DeepLTranslator
  api_key: 'abcdefg0-ffff-bbbb-2222-dasfdasfdasf:fx'
  api_endpoint: 'https://api-free.deepl.com/v2/translate'
  language_fallbacks: { en: en-GB, foo: bar }</pre>
  <pre>
chatgpt:
  label: 'ChatGPT'
  class_name: \Drupal\ai_translate_textfield\ChatGPTTranslator
  api_key: 'abcdefg0-ffff-bbbb-2222-dasfdasfdasf:fx'
  api_endpoint: 'https://api.openai.com/v1/chat/completions'</pre>",
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    try {
      $translatorConfig = Yaml::parse($form_state->getValue('translator_config'));
    }
    catch (ParseException $e) {
      $form_state->setErrorByName('translator_config', $this->t("Couldn't parse the translator configuration Yaml."));
      return;
    }

    if (empty($translatorConfig)) {
      $form_state->setErrorByName('translator_config', $this->t("The translator configuration is empty."));
      return;
    }

    foreach ($translatorConfig as $name => $values) {
      foreach (['label', 'class_name'] as $key) {
        if (empty($values[$key])) {
          $form_state->setErrorByName('translator_config',
            $this->t("You need to have a @key set for service @name", ['@key' => $key, '@name' => $name]));
        }
      }
      if (!class_exists($values['class_name'])) {
        $form_state->setErrorByName('translator_config',
          $this->t("Class @value for @name is not found.", ['@value' => $values['class_name'], '@name' => $name]));
      }

    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    try {
      $translatorConfig = Yaml::parse($form_state->getValue('translator_config'));
    }
    catch (ParseException $e) {
      $this->messenger()->addError($this->t('Error parsing the Yaml for the translator service config: @error', ['@error' => $e->getMessage()]));
      return;
    }

    $this->config('ai_translate_textfield.settings')
      ->set('translator_config', $translatorConfig)
      ->set('button_text', $form_state->getValue('button_text'))
      ->set('strip_tags', $form_state->getValue('strip_tags'))
      ->set('warning_enabled', $form_state->getValue('warning_enabled'))
      ->set('dialog_title', $form_state->getValue('dialog_title'))
      ->set('dialog_content', [
        'value' => $form_state->getValue('dialog_content')['value'],
        'format' => $form_state->getValue('dialog_content')['format'],
      ])
      ->set('dialog_ok_button', $form_state->getValue('dialog_ok_button'))
      ->set('dialog_cancel_button', $form_state->getValue('dialog_cancel_button'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
