<?php

namespace Drupal\ai_upgrade_assistant\Cache;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Collects and manages cache entries for AI Upgrade Assistant.
 *
 * This service helps optimize cache usage by:
 * - Tracking cache hit/miss ratios
 * - Managing cache lifetime based on usage patterns
 * - Implementing intelligent cache pruning
 * - Providing cache statistics for monitoring
 */
class AiUpgradeCacheCollector {

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Cache statistics collection interval.
   *
   * @var int
   */
  const STATS_INTERVAL = 3600;

  /**
   * Constructs a new AiUpgradeCacheCollector.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   The cache tags invalidator.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    CacheBackendInterface $cache,
    CacheTagsInvalidatorInterface $cache_tags_invalidator,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->cache = $cache;
    $this->cacheTagsInvalidator = $cache_tags_invalidator;
    $this->state = $state;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
  }

  /**
   * Records a cache hit.
   *
   * @param string $bin
   *   The cache bin.
   * @param string $cid
   *   The cache ID.
   */
  public function recordHit($bin, $cid) {
    $stats = $this->state->get('ai_upgrade_assistant.cache_stats', []);
    $stats[$bin]['hits'] = ($stats[$bin]['hits'] ?? 0) + 1;
    $stats[$bin]['last_hit'][$cid] = time();
    $this->state->set('ai_upgrade_assistant.cache_stats', $stats);
  }

  /**
   * Records a cache miss.
   *
   * @param string $bin
   *   The cache bin.
   * @param string $cid
   *   The cache ID.
   */
  public function recordMiss($bin, $cid) {
    $stats = $this->state->get('ai_upgrade_assistant.cache_stats', []);
    $stats[$bin]['misses'] = ($stats[$bin]['misses'] ?? 0) + 1;
    $this->state->set('ai_upgrade_assistant.cache_stats', $stats);
  }

  /**
   * Gets cache statistics.
   *
   * @return array
   *   Array of cache statistics.
   */
  public function getStatistics() {
    $stats = $this->state->get('ai_upgrade_assistant.cache_stats', []);
    $processed = [];

    foreach ($stats as $bin => $data) {
      $hits = $data['hits'] ?? 0;
      $misses = $data['misses'] ?? 0;
      $total = $hits + $misses;

      $processed[$bin] = [
        'hits' => $hits,
        'misses' => $misses,
        'hit_ratio' => $total > 0 ? round(($hits / $total) * 100, 2) : 0,
        'last_hit' => $data['last_hit'] ?? [],
      ];
    }

    return $processed;
  }

  /**
   * Prunes unused cache entries.
   *
   * @param int $threshold
   *   Time in seconds after which unused entries are pruned.
   */
  public function pruneUnused($threshold = 604800) {
    $stats = $this->state->get('ai_upgrade_assistant.cache_stats', []);
    $now = time();
    $pruned = 0;

    foreach ($stats as $bin => $data) {
      if (empty($data['last_hit'])) {
        continue;
      }

      foreach ($data['last_hit'] as $cid => $last_hit) {
        if (($now - $last_hit) > $threshold) {
          $this->cache->delete($cid);
          unset($stats[$bin]['last_hit'][$cid]);
          $pruned++;
        }
      }
    }

    $this->state->set('ai_upgrade_assistant.cache_stats', $stats);
    $this->loggerFactory->info('Pruned @count unused cache entries', [
      '@count' => $pruned,
    ]);
  }

  /**
   * Optimizes cache lifetime based on usage patterns.
   */
  public function optimizeLifetime() {
    $stats = $this->getStatistics();
    
    foreach ($stats as $bin => $data) {
      if ($data['hit_ratio'] > 90) {
        // High hit ratio - increase lifetime
        $this->adjustLifetime($bin, 1.5);
      }
      elseif ($data['hit_ratio'] < 50) {
        // Low hit ratio - decrease lifetime
        $this->adjustLifetime($bin, 0.75);
      }
    }
  }

  /**
   * Adjusts cache lifetime for a bin.
   *
   * @param string $bin
   *   The cache bin.
   * @param float $factor
   *   Factor to adjust lifetime by.
   */
  protected function adjustLifetime($bin, $factor) {
    $current = $this->state->get("ai_upgrade_assistant.{$bin}_lifetime", 3600);
    $new = max(300, min(86400, (int) ($current * $factor)));
    $this->state->set("ai_upgrade_assistant.{$bin}_lifetime", $new);
    
    $this->loggerFactory->debug('Adjusted cache lifetime for @bin to @time seconds', [
      '@bin' => $bin,
      '@time' => $new,
    ]);
  }

}
