<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\ai_upgrade_assistant\Service\CommunityLearningService;

/**
 * Controller for the AI Upgrade Assistant forum.
 */
class ForumController extends ControllerBase {

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearningService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new ForumController.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $community_learning_service
   *   The community learning service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    CommunityLearningService $community_learning_service,
    AccountProxyInterface $current_user
  ) {
    $this->communityLearningService = $community_learning_service;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.community_learning'),
      $container->get('current_user')
    );
  }

  /**
   * Displays the forum overview page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   A render array.
   */
  public function overview(Request $request) {
    // Get pagination parameters
    $page = $request->query->get('page', 0);
    $limit = 10;
    $offset = $page * $limit;

    // Get filter parameters
    $filters = [];
    if ($tag = $request->query->get('tag')) {
      $filters['tag'] = $tag;
    }

    // Get topics
    $topics = $this->communityLearningService->getForumTopics($filters, $limit, $offset);

    // Build the page
    $build = [
      '#theme' => 'ai_upgrade_assistant_forum_overview',
      '#topics' => $topics,
      '#filters' => $filters,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/forum'],
      ],
    ];

    // Add create topic button if user has permission
    if ($this->currentUser->hasPermission('create ai_upgrade_assistant forum topics')) {
      $build['actions']['create'] = [
        '#type' => 'link',
        '#title' => $this->t('Create Topic'),
        '#url' => Url::fromRoute('ai_upgrade_assistant.forum.create'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ];
    }

    // Add filters form
    $build['filters'] = [
      '#type' => 'container',
      'tag' => [
        '#type' => 'textfield',
        '#title' => $this->t('Filter by tag'),
        '#value' => $tag,
      ],
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Filter'),
      ],
    ];

    // Add pager
    $build['pager'] = [
      '#type' => 'pager',
      '#quantity' => 5,
    ];

    return $build;
  }

  /**
   * Displays a single forum topic.
   *
   * @param int $topic_id
   *   The topic ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   A render array.
   */
  public function viewTopic($topic_id, Request $request) {
    // Get pagination parameters for replies
    $page = $request->query->get('page', 0);
    $limit = 20;
    $offset = $page * $limit;

    // Get topic and replies
    $topic = $this->communityLearningService->getForumTopics(['id' => $topic_id])[0] ?? NULL;
    if (!$topic) {
      throw new NotFoundHttpException();
    }

    $replies = $this->communityLearningService->getForumReplies($topic_id, $limit, $offset);

    // Build the page
    $build = [
      '#theme' => 'ai_upgrade_assistant_forum_topic',
      '#topic' => $topic,
      '#replies' => $replies,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/forum'],
      ],
    ];

    // Add reply form if user has permission
    if ($this->currentUser->hasPermission('reply to ai_upgrade_assistant forum topics')) {
      $build['reply_form'] = \Drupal::formBuilder()->getForm('Drupal\ai_upgrade_assistant\Form\ForumReplyForm', $topic_id);
    }

    // Add pager for replies
    $build['pager'] = [
      '#type' => 'pager',
      '#quantity' => 5,
    ];

    return $build;
  }

  /**
   * Access callback for viewing a forum topic.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessTopic(AccountProxyInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'view ai_upgrade_assistant forum topics');
  }

  /**
   * Returns forum topics as JSON for AJAX requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function getTopicsJson(Request $request) {
    // Get pagination parameters
    $page = $request->query->get('page', 0);
    $limit = $request->query->get('limit', 10);
    $offset = $page * $limit;

    // Get filter parameters
    $filters = [];
    if ($tag = $request->query->get('tag')) {
      $filters['tag'] = $tag;
    }

    // Get topics
    $topics = $this->communityLearningService->getForumTopics($filters, $limit, $offset);

    return new JsonResponse([
      'topics' => $topics,
      'page' => $page,
      'limit' => $limit,
      'total' => count($topics),
    ]);
  }

}
