<?php

namespace Drupal\ai_upgrade_assistant\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_upgrade_assistant\Service\CommunityLearningService;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Form for creating and editing forum replies.
 */
class ForumReplyForm extends FormBase {

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearningService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new ForumReplyForm.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $community_learning_service
   *   The community learning service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    CommunityLearningService $community_learning_service,
    AccountProxyInterface $current_user,
    MessengerInterface $messenger
  ) {
    $this->communityLearningService = $community_learning_service;
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.community_learning'),
      $container->get('current_user'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_upgrade_assistant_forum_reply_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $topic_id = NULL, $reply_id = NULL) {
    // Load existing reply if editing
    $reply = NULL;
    if ($reply_id) {
      $reply = $this->communityLearningService->getForumReplies($topic_id, 1, 0, ['id' => $reply_id])[0] ?? NULL;
      if (!$reply) {
        throw new NotFoundHttpException();
      }
    }

    $form['content'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Reply'),
      '#required' => TRUE,
      '#format' => 'full_html',
      '#default_value' => $reply ? $reply->content : '',
    ];

    $form['attachments'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Attachments'),
      '#upload_location' => 'public://ai_upgrade_assistant/forum',
      '#multiple' => TRUE,
      '#description' => $this->t('Allowed file types: txt, pdf, zip, tar.gz'),
      '#upload_validators' => [
        'file_validate_extensions' => ['txt pdf zip tar gz'],
        'file_validate_size' => [25 * 1024 * 1024],
      ],
      '#default_value' => $reply ? json_decode($reply->attachments) : [],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $reply ? $this->t('Update Reply') : $this->t('Post Reply'),
    ];

    if ($reply) {
      $form['reply_id'] = [
        '#type' => 'hidden',
        '#value' => $reply_id,
      ];
    }

    $form['topic_id'] = [
      '#type' => 'hidden',
      '#value' => $topic_id,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Prepare attachments
    $attachments = [];
    if ($files = $form_state->getValue('attachments')) {
      foreach ($files as $fid) {
        if ($file = \Drupal::entityTypeManager()->getStorage('file')->load($fid)) {
          $file->setPermanent();
          $file->save();
          $attachments[] = $fid;
        }
      }
    }

    $topic_id = $form_state->getValue('topic_id');
    $reply_data = [
      'content' => $form_state->getValue('content')['value'],
      'attachments' => $attachments,
    ];

    if ($reply_id = $form_state->getValue('reply_id')) {
      // Update existing reply
      $this->communityLearningService->updateForumReply($reply_id, $reply_data);
      $this->messenger->addStatus($this->t('Reply updated.'));
    }
    else {
      // Create new reply
      $this->communityLearningService->addForumReply(
        $topic_id,
        $reply_data['content'],
        $reply_data['attachments']
      );
      $this->messenger->addStatus($this->t('Reply posted.'));
    }

    // Redirect back to the topic page
    $form_state->setRedirect('ai_upgrade_assistant.forum.topic', ['topic_id' => $topic_id]);
  }

}
