<?php

namespace Drupal\ai_upgrade_assistant\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_upgrade_assistant\Service\CommunityLearningService;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Form for deleting forum topics.
 */
class ForumTopicDeleteForm extends ConfirmFormBase {

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearningService;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The topic ID to delete.
   *
   * @var int
   */
  protected $topicId;

  /**
   * Constructs a new ForumTopicDeleteForm.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $community_learning_service
   *   The community learning service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    CommunityLearningService $community_learning_service,
    MessengerInterface $messenger
  ) {
    $this->communityLearningService = $community_learning_service;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.community_learning'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_upgrade_assistant_forum_topic_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $topic_id = NULL) {
    $this->topicId = $topic_id;

    // Load the topic to confirm it exists
    $topic = $this->communityLearningService->getForumTopics(['id' => $topic_id])[0] ?? NULL;
    if (!$topic) {
      throw new NotFoundHttpException();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete this forum topic?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action cannot be undone. All replies to this topic will also be deleted.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('ai_upgrade_assistant.forum.topic', ['topic_id' => $this->topicId]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->communityLearningService->deleteForumTopic($this->topicId);
    $this->messenger->addStatus($this->t('The forum topic has been deleted.'));
    $form_state->setRedirect('ai_upgrade_assistant.forum');
  }

}
