<?php

namespace Drupal\ai_upgrade_assistant\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\State\StateInterface;
use Drupal\ai_upgrade_assistant\Service\EnvironmentDetector;
use Drupal\Core\Url;

/**
 * Configuration form for AI Upgrade Assistant settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The environment detector service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\EnvironmentDetector
   */
  protected $environmentDetector;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\ai_upgrade_assistant\Service\EnvironmentDetector $environment_detector
   *   The environment detector service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    FileSystemInterface $file_system,
    StateInterface $state,
    EnvironmentDetector $environment_detector
  ) {
    parent::__construct($config_factory);
    $this->moduleHandler = $module_handler;
    $this->fileSystem = $file_system;
    $this->state = $state;
    $this->environmentDetector = $environment_detector;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('file_system'),
      $container->get('state'),
      $container->get('ai_upgrade_assistant.environment_detector')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_upgrade_assistant.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_upgrade_assistant_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('ai_upgrade_assistant.settings');

    // Check if we're in a local development environment
    if (!$this->environmentDetector->isLocalDevelopment()) {
      $form['error'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<div class="messages messages--error">ERROR: This appears to be a production environment. This module must only be used in local development environments.</div>'),
      ];
      return $form;
    }

    // Add warning message about local development use only
    $form['warning'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<div class="messages messages--warning">
        <h3>⚠️ LOCAL DEVELOPMENT USE ONLY</h3>
        <p>This module is designed exclusively for local development environments. DO NOT install or use this module on production sites. It makes significant code modifications and requires substantial computational resources that could impact site performance.</p>
        <ul>
          <li>Ensure you are in a local development environment</li>
          <li>Create a complete backup before proceeding</li>
          <li>Never use on production servers</li>
          <li>Review all changes before applying</li>
        </ul>
        </div>'),
      '#weight' => -100,
    ];

    // API Configuration
    $form['api'] = [
      '#type' => 'details',
      '#title' => $this->t('API Configuration'),
      '#open' => TRUE,
    ];

    $form['api']['openai_api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('OpenAI API Key'),
      '#description' => $this->t('Enter your OpenAI API key. This is required for AI-powered analysis.'),
      '#default_value' => $this->state->get('ai_upgrade_assistant.openai_api_key'),
      '#required' => TRUE,
    ];

    $form['api']['ai_model'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Model'),
      '#description' => $this->t('Select the AI model to use for analysis.'),
      '#options' => [
        'gpt-4' => $this->t('GPT-4 (Recommended)'),
        'gpt-3.5-turbo' => $this->t('GPT-3.5 Turbo'),
      ],
      '#default_value' => $config->get('ai_model') ?: 'gpt-4',
    ];

    // OpenAI Settings
    $form['openai_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('OpenAI Settings'),
      '#open' => TRUE,
    ];

    $form['openai_settings']['openai_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAI API Key'),
      '#default_value' => $config->get('openai_api_key'),
      '#required' => TRUE,
      '#description' => $this->t('Enter your OpenAI API key. You can get one from <a href="https://platform.openai.com/account/api-keys">OpenAI\'s website</a>.'),
    ];

    $form['openai_settings']['model_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Model Settings'),
      '#open' => TRUE,
    ];

    $form['openai_settings']['model_settings']['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#options' => [
        'gpt-4' => 'GPT-4',
        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
      ],
      '#default_value' => $config->get('model_settings.model') ?? 'gpt-4',
      '#description' => $this->t('Select the OpenAI model to use for code analysis.'),
    ];

    $form['openai_settings']['model_settings']['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
      '#default_value' => $config->get('model_settings.temperature') ?? 0.7,
      '#description' => $this->t('Controls randomness in the model\'s output. Lower values make the output more focused and deterministic.'),
    ];

    $form['openai_settings']['model_settings']['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max Tokens'),
      '#min' => 100,
      '#max' => 4000,
      '#step' => 100,
      '#default_value' => $config->get('model_settings.max_tokens') ?? 1000,
      '#description' => $this->t('Maximum number of tokens to generate in the response.'),
    ];

    // Analysis Settings
    $form['analysis'] = [
      '#type' => 'details',
      '#title' => $this->t('Analysis Settings'),
      '#open' => TRUE,
    ];

    $form['analysis']['batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch Size'),
      '#description' => $this->t('Number of files to analyze in each batch.'),
      '#default_value' => $config->get('batch_size') ?: 50,
      '#min' => 1,
      '#max' => 100,
    ];

    $form['analysis']['file_patterns'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File Patterns'),
      '#description' => $this->t('Comma-separated list of file patterns to analyze.'),
      '#default_value' => $config->get('file_patterns') ?: '*.php,*.module,*.inc,*.install',
    ];

    $form['analysis']['excluded_paths'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Excluded Paths'),
      '#description' => $this->t('Comma-separated list of paths to exclude from analysis.'),
      '#default_value' => $config->get('excluded_paths') ?: 'vendor/,node_modules/,tests/',
    ];

    // Report Settings
    $form['report'] = [
      '#type' => 'details',
      '#title' => $this->t('Report Settings'),
      '#open' => TRUE,
    ];

    $form['report']['report_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Report Formats'),
      '#description' => $this->t('Select the formats for generated reports.'),
      '#options' => [
        'html' => $this->t('HTML'),
        'json' => $this->t('JSON'),
        'markdown' => $this->t('Markdown'),
      ],
      '#default_value' => $config->get('report_formats') ?: ['html', 'json'],
    ];

    $form['report']['report_directory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Report Directory'),
      '#description' => $this->t('Directory where reports will be saved.'),
      '#default_value' => $config->get('report_directory') ?: 'public://ai-upgrade-reports',
    ];

    // GitHub Settings
    $form['github'] = [
      '#type' => 'details',
      '#title' => $this->t('GitHub Settings'),
      '#open' => TRUE,
    ];

    $form['github']['github_token'] = [
      '#type' => 'password',
      '#title' => $this->t('GitHub Personal Access Token'),
      '#description' => $this->t('Enter your GitHub personal access token to increase API rate limits. You can create one at @url', [
        '@url' => 'https://github.com/settings/tokens',
      ]),
      '#default_value' => $this->state->get('ai_upgrade_assistant.github_token'),
    ];

    // Public Dataset Settings
    $form['dataset'] = [
      '#type' => 'details',
      '#title' => $this->t('Public Dataset Settings'),
      '#open' => TRUE,
      '#description' => $this->t('All shared patterns are publicly available on HuggingFace at @url. This transparency helps build trust and allows the community to validate and improve the upgrade patterns.', [
        '@url' => 'https://huggingface.co/datasets/drupal/ai-upgrade-patterns',
      ]),
    ];

    $form['dataset']['huggingface_token'] = [
      '#type' => 'password',
      '#title' => $this->t('HuggingFace API Token'),
      '#description' => $this->t('Get your token from HuggingFace Settings. This is only used for contributing patterns - anyone can view the dataset without a token.'),
      '#default_value' => $this->state->get('ai_upgrade_assistant.huggingface_token'),
    ];

    // Pattern Preview
    $form['dataset']['pattern_preview'] = [
      '#type' => 'details',
      '#title' => $this->t('Pattern Sharing Preview'),
      '#description' => $this->t('See exactly what data will be shared publicly. All patterns are anonymized before sharing.'),
      '#open' => FALSE,
    ];

    $patterns = $this->getPatternPreview();
    if (empty($patterns)) {
      $form['dataset']['pattern_preview']['content'] = [
        '#markup' => $this->t('No patterns available for preview.'),
      ];
    }
    else {
      $form['dataset']['pattern_preview']['content'] = [
        '#theme' => 'table',
        '#header' => ['Type', 'Before', 'After', 'Metadata'],
        '#rows' => $patterns,
      ];
    }

    // Dataset Statistics
    $form['dataset']['statistics'] = [
      '#type' => 'details',
      '#title' => $this->t('Public Dataset Statistics'),
      '#description' => $this->t('Real-time statistics from our public dataset:'),
      '#open' => FALSE,
    ];

    $stats = $this->getDatasetStatistics();
    if (!empty($stats)) {
      $form['dataset']['statistics']['content'] = [
        '#theme' => 'item_list',
        '#items' => $stats,
      ];
    }

    // Advanced Settings
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['parallel_processing'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Parallel Processing'),
      '#description' => $this->t('Use multiple processes to speed up analysis. May increase memory usage.'),
      '#default_value' => $config->get('parallel_processing') ?: FALSE,
    ];

    $form['advanced']['cache_ttl'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache TTL'),
      '#description' => $this->t('Time in seconds to cache API responses and analysis results.'),
      '#default_value' => $config->get('cache_ttl') ?: 3600,
      '#min' => 0,
    ];

    $form['advanced']['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug Mode'),
      '#description' => $this->t('Enable detailed logging and debugging information.'),
      '#default_value' => $config->get('debug_mode') ?: FALSE,
    ];

    $form['advanced']['mock_api'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Mock API'),
      '#description' => $this->t('If enabled, will use mock results when API calls fail. Use for testing only.'),
      '#default_value' => $config->get('mock_api') ?: FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate report directory
    $report_dir = $form_state->getValue('report_directory');
    if (!$this->fileSystem->prepareDirectory($report_dir, FileSystemInterface::CREATE_DIRECTORY)) {
      $form_state->setError($form['report']['report_directory'], $this->t('Unable to create or access report directory.'));
    }

    // Validate batch size
    $batch_size = $form_state->getValue('batch_size');
    if ($batch_size < 1 || $batch_size > 100) {
      $form_state->setError($form['analysis']['batch_size'], $this->t('Batch size must be between 1 and 100.'));
    }

    // Validate API key if provided
    $api_key = $form_state->getValue('openai_api_key');
    if (!empty($api_key) && !$this->validateApiKey($api_key)) {
      $form_state->setError($form['api']['openai_api_key'], $this->t('Invalid OpenAI API key format.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_upgrade_assistant.settings');
    
    // Save sensitive data to state
    $this->state->set('ai_upgrade_assistant.openai_api_key', $form_state->getValue('openai_api_key'));
    $this->state->set('ai_upgrade_assistant.github_token', $form_state->getValue('github_token'));
    $this->state->set('ai_upgrade_assistant.huggingface_token', $form_state->getValue('huggingface_token'));

    // Save non-sensitive config
    $fields = [
      'ai_model',
      'batch_size',
      'file_patterns',
      'excluded_paths',
      'report_formats',
      'report_directory',
      'parallel_processing',
      'cache_ttl',
      'debug_mode',
      'mock_api',
      'openai_api_key',
      'model_settings',
    ];

    foreach ($fields as $field) {
      $config->set($field, $form_state->getValue($field));
    }

    $config->save();
  }

  /**
   * Gets a preview of patterns that will be shared publicly.
   *
   * @return array
   *   Array of pattern preview data.
   */
  protected function getPatternPreview() {
    // Implementation will be added in a separate PR
    return [];
  }

  /**
   * Gets statistics from the public dataset.
   *
   * @return array
   *   Array of dataset statistics.
   */
  protected function getDatasetStatistics() {
    // Implementation will be added in a separate PR
    return [];
  }

  /**
   * Validates an OpenAI API key format.
   *
   * @param string $api_key
   *   The API key to validate.
   *
   * @return bool
   *   TRUE if the API key format is valid, FALSE otherwise.
   */
  protected function validateApiKey($api_key) {
    return (bool) preg_match('/^sk-[a-zA-Z0-9]{32,}$/', $api_key);
  }
}
