<?php

namespace Drupal\ai_upgrade_assistant\Plugin\QueueWorker;

use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\ai_upgrade_assistant\Service\UpdateHistoryService;
use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzerService;

/**
 * Processes queued module updates.
 *
 * @QueueWorker(
 *   id = "ai_upgrade_assistant_updates",
 *   title = @Translation("AI Upgrade Assistant Update Processor"),
 *   cron = {"time" = 60}
 * )
 */
class UpdateProcessor extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The update history service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\UpdateHistoryService
   */
  protected $history;

  /**
   * The project analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzerService
   */
  protected $analyzer;

  /**
   * Constructs a new UpdateProcessor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\ai_upgrade_assistant\Service\UpdateHistoryService $history
   *   The update history service.
   * @param \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzerService $analyzer
   *   The project analyzer service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    UpdateHistoryService $history,
    ProjectAnalyzerService $analyzer
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
    $this->logger = $logger_factory->get('ai_upgrade_assistant');
    $this->history = $history;
    $this->analyzer = $analyzer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('state'),
      $container->get('logger.factory'),
      $container->get('ai_upgrade_assistant.update_history'),
      $container->get('ai_upgrade_assistant.project_analyzer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $update = $data['update'];
    $update_id = $data['update_id'];
    $module_name = $update['module_name'];

    try {
      $this->logger->info(
        'Starting update process for @module from @from to @to',
        [
          '@module' => $module_name,
          '@from' => $update['from_version'],
          '@to' => $update['to_version'],
        ]
      );

      // Analyze update complexity and generate patches
      $analysis = $this->analyzer->analyzeUpdate($module_name, $update['from_version'], $update['to_version']);
      
      // Record patches
      if (!empty($analysis['patches'])) {
        $this->history->recordPatches($update_id, $analysis['patches']);
      }

      // Apply patches and perform update
      $success = $this->applyUpdate($update, $analysis);

      // Record completion
      $results = [
        'analysis' => $analysis,
        'duration' => time() - $update['start_time'],
      ];

      $this->history->recordUpdateCompletion(
        $update_id,
        $success ? 'success' : 'failure',
        $results
      );

      // Clear in-progress state
      $this->clearInProgressState($update);

      $this->logger->info(
        'Completed update process for @module to @version with status: @status',
        [
          '@module' => $module_name,
          '@version' => $update['to_version'],
          '@status' => $success ? 'success' : 'failure',
        ]
      );

    }
    catch (\Exception $e) {
      // Record error
      $this->history->recordErrors($update_id, [[
        'type' => get_class($e),
        'message' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]]);

      // Record failure
      $this->history->recordUpdateCompletion($update_id, 'failure', [
        'error' => $e->getMessage(),
      ]);

      // Clear in-progress state
      $this->clearInProgressState($update);

      $this->logger->error(
        'Error processing update for @module: @error',
        [
          '@module' => $module_name,
          '@error' => $e->getMessage(),
        ]
      );

      // Re-throw the exception to let the queue system handle it
      throw $e;
    }
  }

  /**
   * Applies the update using generated patches.
   *
   * @param array $update
   *   The update information.
   * @param array $analysis
   *   The update analysis results.
   *
   * @return bool
   *   TRUE if update was successful, FALSE otherwise.
   */
  protected function applyUpdate(array $update, array $analysis) {
    // This is where we would implement the actual update process
    // For now, we'll just return true as a placeholder
    return TRUE;
  }

  /**
   * Clears the in-progress state for an update.
   *
   * @param array $update
   *   The update information.
   */
  protected function clearInProgressState(array $update) {
    $in_progress = $this->state->get('ai_upgrade_assistant.updates_in_progress', []);
    $key = $update['module_name'] . '-' . $update['to_version'];
    unset($in_progress[$key]);
    $this->state->set('ai_upgrade_assistant.updates_in_progress', $in_progress);
  }

}
