<?php

namespace Drupal\ai_upgrade_assistant\Service\NodeVisitor;

use PhpParser\Node;
use PhpParser\NodeVisitorAbstract;

/**
 * Node visitor that detects deprecated constant usage.
 */
class DeprecatedConstantVisitor extends NodeVisitorAbstract {

  /**
   * List of findings.
   *
   * @var array
   */
  protected $findings = [];

  /**
   * List of deprecated constants and their replacements.
   *
   * @var array
   */
  protected $deprecatedConstants = [
    // Core constants
    'DRUPAL_ROOT' => [
      'replacement' => 'app.root service parameter',
      'version' => '10.0.0',
      'critical' => false,
      'note' => 'Use dependency injection instead',
    ],
    'REQUEST_TIME' => [
      'replacement' => '\Drupal::time()->getRequestTime()',
      'version' => '10.0.0',
      'critical' => false,
    ],
    // Language constants
    'LANGUAGE_NONE' => [
      'replacement' => 'LanguageInterface::LANGCODE_NOT_SPECIFIED',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'LANGUAGE_NOT_SPECIFIED' => [
      'replacement' => 'LanguageInterface::LANGCODE_NOT_SPECIFIED',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // File constants
    'FILE_STATUS_PERMANENT' => [
      'replacement' => 'FileInterface::STATUS_PERMANENT',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'FILE_STATUS_TEMPORARY' => [
      'replacement' => 'FileInterface::STATUS_TEMPORARY',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Node constants
    'NODE_NOT_PUBLISHED' => [
      'replacement' => 'NodeInterface::NOT_PUBLISHED',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'NODE_PUBLISHED' => [
      'replacement' => 'NodeInterface::PUBLISHED',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // User constants
    'USER_REGISTER_ADMINISTRATORS_ONLY' => [
      'replacement' => 'UserInterface::REGISTER_ADMINISTRATORS_ONLY',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'USER_REGISTER_VISITORS' => [
      'replacement' => 'UserInterface::REGISTER_VISITORS',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Cache constants
    'CACHE_PERMANENT' => [
      'replacement' => 'Cache::PERMANENT',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'CACHE_TEMPORARY' => [
      'replacement' => 'Cache::TEMPORARY',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Menu constants
    'MENU_NORMAL_ITEM' => [
      'replacement' => 'MenuLinkInterface::NORMAL_ITEM',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'MENU_CALLBACK' => [
      'replacement' => 'MenuLinkInterface::CALLBACK_ITEM',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Update constants
    'UPDATE_NOT_SECURE' => [
      'replacement' => 'UpdateManagerInterface::NOT_SECURE',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'UPDATE_NOT_CURRENT' => [
      'replacement' => 'UpdateManagerInterface::NOT_CURRENT',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Batch constants
    'BATCH_API_TRANSPARENT' => [
      'replacement' => 'BatchInterface::BATCH_PROCESSING_IN_BACKGROUND',
      'version' => '10.0.0',
      'critical' => false,
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function enterNode(Node $node) {
    if ($node instanceof Node\Expr\ConstFetch) {
      $name = $node->name->toString();
      if (isset($this->deprecatedConstants[$name])) {
        $finding = [
          'type' => 'deprecated_constant',
          'name' => $name,
          'line' => $node->getLine(),
          'file' => $node->getAttribute('file'),
          'replacement' => $this->deprecatedConstants[$name]['replacement'],
          'version' => $this->deprecatedConstants[$name]['version'],
          'critical' => $this->deprecatedConstants[$name]['critical'],
        ];

        if (isset($this->deprecatedConstants[$name]['note'])) {
          $finding['note'] = $this->deprecatedConstants[$name]['note'];
        }

        // Check usage context
        $parent = $node->getAttribute('parent');
        if ($parent) {
          if ($parent instanceof Node\Expr\BinaryOp) {
            $finding['context'] = 'comparison';
          }
          elseif ($parent instanceof Node\Expr\Assign) {
            $finding['context'] = 'assignment';
          }
          elseif ($parent instanceof Node\Arg) {
            $finding['context'] = 'function_argument';
          }
        }

        $this->findings[] = $finding;
      }
    }
  }

  /**
   * Gets the findings.
   *
   * @return array
   *   Array of findings.
   */
  public function getFindings() {
    return $this->findings;
  }

  /**
   * Resets the findings.
   */
  public function reset() {
    $this->findings = [];
  }

}
