<?php

namespace Drupal\ai_upgrade_assistant\Service\NodeVisitor;

use PhpParser\Node;
use PhpParser\NodeVisitorAbstract;

/**
 * Node visitor that detects deprecated function calls.
 */
class DeprecatedFunctionVisitor extends NodeVisitorAbstract {

  /**
   * List of findings.
   *
   * @var array
   */
  protected $findings = [];

  /**
   * List of deprecated functions and their replacements.
   *
   * @var array
   */
  protected $deprecatedFunctions = [
    // Core functions
    'drupal_set_message' => [
      'replacement' => '\Drupal::messenger()->addMessage()',
      'version' => '8.5.0',
      'critical' => true,
    ],
    'format_date' => [
      'replacement' => '\Drupal::service("date.formatter")->format()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Entity loading
    'node_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("node")->load()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'user_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("user")->load()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'file_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("file")->load()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'entity_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage($entity_type)->load()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Database
    'db_query' => [
      'replacement' => '\Drupal::database()->query()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'db_select' => [
      'replacement' => '\Drupal::database()->select()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // File system
    'file_unmanaged_copy' => [
      'replacement' => '\Drupal::service("file_system")->copy()',
      'version' => '8.7.0',
      'critical' => true,
    ],
    'file_unmanaged_delete' => [
      'replacement' => '\Drupal::service("file_system")->delete()',
      'version' => '8.7.0',
      'critical' => true,
    ],
    'file_unmanaged_move' => [
      'replacement' => '\Drupal::service("file_system")->move()',
      'version' => '8.7.0',
      'critical' => true,
    ],
    // Cache
    'cache_get' => [
      'replacement' => '\Drupal::cache()->get()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'cache_set' => [
      'replacement' => '\Drupal::cache()->set()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // URL/path handling
    'drupal_get_path' => [
      'replacement' => '\Drupal::service("extension.list.$type")->getPath()',
      'version' => '9.3.0',
      'critical' => true,
    ],
    'drupal_get_filename' => [
      'replacement' => '\Drupal::service("extension.list.$type")->getPathname()',
      'version' => '9.3.0',
      'critical' => true,
    ],
    // Form API
    'drupal_get_form' => [
      'replacement' => '\Drupal::formBuilder()->getForm()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'drupal_build_form' => [
      'replacement' => '\Drupal::formBuilder()->buildForm()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Theme
    'theme' => [
      'replacement' => '\Drupal::theme()->render()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Language
    'language_default' => [
      'replacement' => '\Drupal::languageManager()->getDefaultLanguage()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // State system
    'variable_get' => [
      'replacement' => '\Drupal::state()->get()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    'variable_set' => [
      'replacement' => '\Drupal::state()->set()',
      'version' => '8.0.0',
      'critical' => true,
    ],
    // Batch API
    'batch_set' => [
      'replacement' => '$batch = new Batch(); $batch->set()',
      'version' => '10.0.0',
      'critical' => false,
    ],
    // Token handling
    'token_replace' => [
      'replacement' => '\Drupal::token()->replace()',
      'version' => '8.0.0',
      'critical' => true,
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function leaveNode(Node $node) {
    if ($node instanceof Node\Expr\FuncCall && $node->name instanceof Node\Name) {
      $name = $node->name->toString();
      if (isset($this->deprecatedFunctions[$name])) {
        $this->findings[] = [
          'type' => 'deprecated_function',
          'name' => $name,
          'line' => $node->getLine(),
          'replacement' => $this->deprecatedFunctions[$name]['replacement'],
          'version' => $this->deprecatedFunctions[$name]['version'],
          'critical' => $this->deprecatedFunctions[$name]['critical'],
          'file' => $node->getAttribute('file'),
          'args_count' => count($node->args),
        ];

        // Add context about the function call
        if ($node->args) {
          $args = [];
          foreach ($node->args as $arg) {
            if ($arg->value instanceof Node\Scalar\String_) {
              $args[] = $arg->value->value;
            }
          }
          if ($args) {
            $this->findings[count($this->findings) - 1]['arg_values'] = $args;
          }
        }
      }
    }
    return null;
  }

  /**
   * Gets the findings.
   *
   * @return array
   *   Array of findings.
   */
  public function getFindings() {
    return $this->findings;
  }

  /**
   * Resets the findings.
   */
  public function reset() {
    $this->findings = [];
  }

}
