<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Template\Attribute;

/**
 * Service for preprocessing theme variables.
 *
 * This service handles the preprocessing of theme variables for the AI Upgrade
 * Assistant module, ensuring consistent styling and dynamic content handling
 * across all themed output.
 */
class ThemePreprocessor {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * Constructs a new ThemePreprocessor object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ThemeManagerInterface $theme_manager
  ) {
    $this->configFactory = $config_factory;
    $this->themeManager = $theme_manager;
  }

  /**
   * Preprocesses variables for the upgrade status display.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessUpgradeStatus(array &$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');

    // Add wrapper attributes.
    $variables['wrapper_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-status',
        'ai-upgrade-status--' . $variables['status'],
      ],
    ]);

    // Add header attributes.
    $variables['header_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-status__header',
      ],
    ]);

    // Add content attributes.
    $variables['content_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-status__content',
      ],
    ]);

    // Add theme-specific classes if needed.
    $activeTheme = $this->themeManager->getActiveTheme();
    if ($activeTheme->getName() === 'claro' || $activeTheme->getName() === 'gin') {
      $variables['wrapper_attributes']->addClass('ai-upgrade-status--admin');
    }

    // Add status-specific classes and icons.
    switch ($variables['status']) {
      case 'success':
        $variables['icon'] = 'check';
        $variables['status_class'] = 'status--success';
        break;

      case 'warning':
        $variables['icon'] = 'warning';
        $variables['status_class'] = 'status--warning';
        break;

      case 'error':
        $variables['icon'] = 'error';
        $variables['status_class'] = 'status--error';
        break;

      default:
        $variables['icon'] = 'info';
        $variables['status_class'] = 'status--info';
    }

    // Add accessibility attributes.
    $variables['wrapper_attributes']->setAttribute('role', 'status');
    $variables['wrapper_attributes']->setAttribute('aria-label', $this->t('Upgrade Status: @status', [
      '@status' => ucfirst($variables['status']),
    ]));
  }

  /**
   * Preprocesses variables for the upgrade analysis display.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessUpgradeAnalysis(array &$variables) {
    // Add wrapper attributes.
    $variables['wrapper_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-analysis',
      ],
    ]);

    // Add section attributes for each analysis section.
    foreach ($variables['sections'] as &$section) {
      $section['attributes'] = new Attribute([
        'class' => [
          'ai-upgrade-analysis__section',
          'ai-upgrade-analysis__section--' . $section['type'],
        ],
      ]);

      // Add specific classes based on section type.
      switch ($section['type']) {
        case 'deprecated':
          $section['icon'] = 'warning';
          $section['priority_class'] = 'priority--high';
          break;

        case 'security':
          $section['icon'] = 'shield';
          $section['priority_class'] = 'priority--critical';
          break;

        case 'performance':
          $section['icon'] = 'speed';
          $section['priority_class'] = 'priority--medium';
          break;
      }
    }

    // Add chart container attributes if charts are present.
    if (!empty($variables['charts'])) {
      $variables['chart_attributes'] = new Attribute([
        'class' => [
          'ai-upgrade-analysis__charts',
        ],
      ]);
    }
  }

  /**
   * Preprocesses variables for the upgrade progress display.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessUpgradeProgress(array &$variables) {
    // Add wrapper attributes.
    $variables['wrapper_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-progress',
      ],
    ]);

    // Calculate and add progress percentage.
    if (isset($variables['completed']) && isset($variables['total'])) {
      $percentage = ($variables['completed'] / $variables['total']) * 100;
      $variables['progress_percentage'] = round($percentage);

      // Add progress bar attributes.
      $variables['progress_attributes'] = new Attribute([
        'class' => [
          'ai-upgrade-progress__bar',
        ],
        'style' => 'width: ' . $percentage . '%;',
      ]);
    }

    // Add status classes based on progress.
    if ($percentage === 100) {
      $variables['status_class'] = 'status--complete';
    }
    elseif ($percentage >= 75) {
      $variables['status_class'] = 'status--almost-complete';
    }
    elseif ($percentage >= 25) {
      $variables['status_class'] = 'status--in-progress';
    }
    else {
      $variables['status_class'] = 'status--starting';
    }
  }

}
