<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\ai_upgrade_assistant\Service\MachineLearning\PatternLearningManager;

/**
 * Service for generating optimal upgrade paths.
 */
class UpgradePathGenerator {

  /**
   * The pattern learning manager.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\MachineLearning\PatternLearningManager
   */
  protected $patternLearning;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Cache of upgrade paths.
   *
   * @var array
   */
  protected $pathCache = [];

  /**
   * Constructs a new UpgradePathGenerator.
   */
  public function __construct(
    PatternLearningManager $pattern_learning,
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->patternLearning = $pattern_learning;
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Generates an optimal upgrade path for a module.
   *
   * @param string $module_name
   *   The name of the module.
   * @param string $current_version
   *   Current version of Drupal core.
   * @param string $target_version
   *   Target version of Drupal core.
   *
   * @return array
   *   The generated upgrade path.
   */
  public function generateUpgradePath($module_name, $current_version, $target_version) {
    $cache_key = "{$module_name}:{$current_version}:{$target_version}";
    if (isset($this->pathCache[$cache_key])) {
      return $this->pathCache[$cache_key];
    }

    try {
      // Initialize upgrade path
      $path = [
        'module' => $module_name,
        'current_version' => $current_version,
        'target_version' => $target_version,
        'steps' => [],
        'dependencies' => [],
        'risks' => [],
        'estimated_effort' => 0,
      ];

      // Get module info
      $module = $this->moduleHandler->getModule($module_name);
      $module_path = $module->getPath();

      // Analyze version-specific changes
      $version_changes = $this->analyzeVersionChanges($current_version, $target_version);

      // Get module dependencies
      $dependencies = $this->analyzeDependencies($module_name);

      // Generate steps based on patterns and version changes
      $steps = $this->generateUpgradeSteps($module_name, $version_changes);

      // Optimize the upgrade path
      $optimized_steps = $this->optimizeUpgradePath($steps);

      // Calculate risks and effort
      $risk_assessment = $this->assessRisks($module_name, $optimized_steps);

      // Build the final upgrade path
      $path['steps'] = $optimized_steps;
      $path['dependencies'] = $dependencies;
      $path['risks'] = $risk_assessment['risks'];
      $path['estimated_effort'] = $risk_assessment['effort'];
      $path['metadata'] = [
        'generated_at' => time(),
        'confidence_score' => $this->calculateConfidenceScore($optimized_steps),
      ];

      // Cache the path
      $this->pathCache[$cache_key] = $path;

      return $path;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Failed to generate upgrade path for @module: @error',
        [
          '@module' => $module_name,
          '@error' => $e->getMessage(),
        ]
      );
      throw $e;
    }
  }

  /**
   * Analyzes version-specific changes between Drupal versions.
   */
  protected function analyzeVersionChanges($current_version, $target_version) {
    $changes = [];
    $versions = $this->getVersionsBetween($current_version, $target_version);

    foreach ($versions as $version) {
      $changes[$version] = [
        'api_changes' => $this->getApiChanges($version),
        'deprecated_features' => $this->getDeprecatedFeatures($version),
        'new_features' => $this->getNewFeatures($version),
        'breaking_changes' => $this->getBreakingChanges($version),
      ];
    }

    return $changes;
  }

  /**
   * Gets all versions between two Drupal versions.
   */
  protected function getVersionsBetween($start, $end) {
    $versions = [];
    $major_start = (int) substr($start, 0, strpos($start, '.'));
    $major_end = (int) substr($end, 0, strpos($end, '.'));

    for ($major = $major_start; $major <= $major_end; $major++) {
      $minor_versions = $this->getMinorVersions($major);
      foreach ($minor_versions as $version) {
        if (version_compare($version, $start, '>=') && 
            version_compare($version, $end, '<=')) {
          $versions[] = $version;
        }
      }
    }

    return $versions;
  }

  /**
   * Gets minor versions for a major Drupal version.
   */
  protected function getMinorVersions($major) {
    // This could be expanded to fetch from drupal.org API
    $versions = [];
    switch ($major) {
      case 8:
        $versions = ['8.8.0', '8.9.0'];
        break;

      case 9:
        $versions = ['9.0.0', '9.1.0', '9.2.0', '9.3.0', '9.4.0', '9.5.0'];
        break;

      case 10:
        $versions = ['10.0.0', '10.1.0'];
        break;
    }
    return $versions;
  }

  /**
   * Gets API changes for a specific version.
   */
  protected function getApiChanges($version) {
    // This could be expanded to fetch from drupal.org API
    $changes = [];
    
    // Example API changes
    $api_changes = [
      '8.8.0' => [
        ['type' => 'method', 'old' => 'url()', 'new' => 'toUrl()', 'class' => 'EntityInterface'],
      ],
      '9.0.0' => [
        ['type' => 'service', 'old' => 'entity.manager', 'new' => 'entity_type.manager'],
      ],
      '10.0.0' => [
        ['type' => 'function', 'old' => 'drupal_set_message()', 'new' => 'messenger()->addMessage()'],
      ],
    ];

    return $api_changes[$version] ?? [];
  }

  /**
   * Gets deprecated features for a specific version.
   */
  protected function getDeprecatedFeatures($version) {
    // This could be expanded to fetch from drupal.org API
    $deprecated = [
      '8.8.0' => [
        'drupal_set_message',
        '\Drupal::entityManager',
      ],
      '9.0.0' => [
        'SafeMarkup::checkPlain',
        'drupal_render',
      ],
      '10.0.0' => [
        'file_scan_directory',
        'file_prepare_directory',
      ],
    ];

    return $deprecated[$version] ?? [];
  }

  /**
   * Gets new features for a specific version.
   */
  protected function getNewFeatures($version) {
    // This could be expanded to fetch from drupal.org API
    $features = [
      '8.8.0' => [
        'Media Library',
        'Layout Builder',
      ],
      '9.0.0' => [
        'Olivero theme',
        'Claro admin theme',
      ],
      '10.0.0' => [
        'Decoupled menus',
        'JavaScript modernization',
      ],
    ];

    return $features[$version] ?? [];
  }

  /**
   * Gets breaking changes for a specific version.
   */
  protected function getBreakingChanges($version) {
    // This could be expanded to fetch from drupal.org API
    $changes = [
      '8.8.0' => [
        'Removed deprecated code from core',
        'Updated Symfony dependencies',
      ],
      '9.0.0' => [
        'Removed deprecated APIs',
        'Updated minimum PHP version',
      ],
      '10.0.0' => [
        'Removed jQuery UI',
        'Updated minimum PHP version',
      ],
    ];

    return $changes[$version] ?? [];
  }

  /**
   * Analyzes module dependencies.
   */
  protected function analyzeDependencies($module_name) {
    $module = $this->moduleHandler->getModule($module_name);
    $info = $module->info;

    $dependencies = [
      'required' => [],
      'optional' => [],
      'conflicts' => [],
    ];

    if (!empty($info['dependencies'])) {
      foreach ($info['dependencies'] as $dependency) {
        $dependencies['required'][] = $this->parseDependency($dependency);
      }
    }

    return $dependencies;
  }

  /**
   * Parses a dependency string.
   */
  protected function parseDependency($dependency) {
    // Format: module_name (>= version)
    if (preg_match('/^([a-z0-9_]+)\s*(?:\((.*?)\))?/', $dependency, $matches)) {
      return [
        'name' => $matches[1],
        'constraint' => isset($matches[2]) ? $matches[2] : null,
      ];
    }
    return ['name' => $dependency];
  }

  /**
   * Generates upgrade steps based on patterns and version changes.
   */
  protected function generateUpgradeSteps($module_name, array $version_changes) {
    $steps = [];

    foreach ($version_changes as $version => $changes) {
      // Add version-specific steps
      $steps = array_merge($steps, $this->generateVersionSteps($version, $changes));

      // Add pattern-based steps
      $pattern_steps = $this->generatePatternSteps($module_name, $version);
      $steps = array_merge($steps, $pattern_steps);
    }

    return $steps;
  }

  /**
   * Generates steps for a specific version.
   */
  protected function generateVersionSteps($version, array $changes) {
    $steps = [];

    // Handle API changes
    foreach ($changes['api_changes'] as $change) {
      $steps[] = [
        'type' => 'api_update',
        'version' => $version,
        'description' => sprintf(
          'Update %s from %s to %s',
          $change['type'],
          $change['old'],
          $change['new']
        ),
        'old' => $change['old'],
        'new' => $change['new'],
        'automated' => true,
      ];
    }

    // Handle deprecated features
    foreach ($changes['deprecated_features'] as $feature) {
      $steps[] = [
        'type' => 'deprecation',
        'version' => $version,
        'description' => "Remove deprecated feature: {$feature}",
        'feature' => $feature,
        'automated' => true,
      ];
    }

    // Handle breaking changes
    foreach ($changes['breaking_changes'] as $change) {
      $steps[] = [
        'type' => 'breaking_change',
        'version' => $version,
        'description' => $change,
        'automated' => false,
      ];
    }

    return $steps;
  }

  /**
   * Generates steps based on learned patterns.
   */
  protected function generatePatternSteps($module_name, $version) {
    $steps = [];
    
    // Get patterns from the module
    $patterns = $this->patternLearning->getModulePatterns($module_name);

    foreach ($patterns as $pattern) {
      // Get transformation prediction
      $prediction = $this->patternLearning->predictTransformation($pattern);
      
      if ($prediction && $prediction['confidence'] > 0.8) {
        $steps[] = [
          'type' => 'pattern_transformation',
          'version' => $version,
          'description' => $this->generateStepDescription($prediction),
          'pattern' => $pattern,
          'transformation' => $prediction['transformation'],
          'confidence' => $prediction['confidence'],
          'automated' => $prediction['confidence'] > 0.95,
        ];
      }
    }

    return $steps;
  }

  /**
   * Optimizes the upgrade path.
   */
  protected function optimizeUpgradePath(array $steps) {
    // Sort steps by dependencies
    $steps = $this->sortStepsByDependencies($steps);

    // Group related steps
    $steps = $this->groupRelatedSteps($steps);

    // Prioritize steps
    $steps = $this->prioritizeSteps($steps);

    return $steps;
  }

  /**
   * Sorts steps by dependencies.
   */
  protected function sortStepsByDependencies(array $steps) {
    $sorted = [];
    $visited = [];

    foreach ($steps as $step) {
      $this->sortStep($step, $sorted, $visited, $steps);
    }

    return $sorted;
  }

  /**
   * Helper function for dependency sorting.
   */
  protected function sortStep($step, &$sorted, &$visited, $steps) {
    $id = $this->getStepId($step);

    if (isset($visited[$id])) {
      return;
    }

    $visited[$id] = true;

    if (isset($step['dependencies'])) {
      foreach ($step['dependencies'] as $dep) {
        foreach ($steps as $dep_step) {
          if ($this->getStepId($dep_step) === $dep) {
            $this->sortStep($dep_step, $sorted, $visited, $steps);
          }
        }
      }
    }

    $sorted[] = $step;
  }

  /**
   * Gets a unique identifier for a step.
   */
  protected function getStepId($step) {
    return md5(serialize($step));
  }

  /**
   * Groups related steps together.
   */
  protected function groupRelatedSteps(array $steps) {
    $groups = [];
    $current_group = [];

    foreach ($steps as $step) {
      if (empty($current_group)) {
        $current_group[] = $step;
        continue;
      }

      $last_step = end($current_group);
      if ($this->areStepsRelated($last_step, $step)) {
        $current_group[] = $step;
      }
      else {
        $groups[] = $current_group;
        $current_group = [$step];
      }
    }

    if (!empty($current_group)) {
      $groups[] = $current_group;
    }

    return array_merge(...$groups);
  }

  /**
   * Checks if two steps are related.
   */
  protected function areStepsRelated($step1, $step2) {
    // Steps are related if they:
    // 1. Are of the same type
    if ($step1['type'] === $step2['type']) {
      return true;
    }

    // 2. Affect the same file
    if (isset($step1['file']) && isset($step2['file']) && 
        $step1['file'] === $step2['file']) {
      return true;
    }

    // 3. Are part of the same version upgrade
    if (isset($step1['version']) && isset($step2['version']) && 
        $step1['version'] === $step2['version']) {
      return true;
    }

    return false;
  }

  /**
   * Prioritizes steps based on importance and risk.
   */
  protected function prioritizeSteps(array $steps) {
    $priority_map = [
      'breaking_change' => 1,
      'api_update' => 2,
      'deprecation' => 3,
      'pattern_transformation' => 4,
    ];

    usort($steps, function ($a, $b) use ($priority_map) {
      $priority_a = $priority_map[$a['type']] ?? 999;
      $priority_b = $priority_map[$b['type']] ?? 999;

      if ($priority_a === $priority_b) {
        // If same priority, sort by confidence (if available)
        $conf_a = $a['confidence'] ?? 0;
        $conf_b = $b['confidence'] ?? 0;
        return $conf_b <=> $conf_a;
      }

      return $priority_a <=> $priority_b;
    });

    return $steps;
  }

  /**
   * Assesses risks for the upgrade path.
   */
  protected function assessRisks($module_name, array $steps) {
    $risks = [];
    $total_effort = 0;

    foreach ($steps as $step) {
      $risk = $this->calculateStepRisk($step);
      if ($risk['level'] !== 'low') {
        $risks[] = $risk;
      }
      $total_effort += $risk['effort'];
    }

    return [
      'risks' => $risks,
      'effort' => $total_effort,
    ];
  }

  /**
   * Calculates risk for a single step.
   */
  protected function calculateStepRisk($step) {
    $risk = [
      'step' => $step['description'],
      'level' => 'low',
      'effort' => 1,
      'mitigation' => [],
    ];

    switch ($step['type']) {
      case 'breaking_change':
        $risk['level'] = 'high';
        $risk['effort'] = 5;
        $risk['mitigation'][] = 'Comprehensive testing required';
        $risk['mitigation'][] = 'Create backup before implementing';
        break;

      case 'api_update':
        $risk['level'] = 'medium';
        $risk['effort'] = 3;
        $risk['mitigation'][] = 'Update all API references';
        $risk['mitigation'][] = 'Test affected functionality';
        break;

      case 'deprecation':
        $risk['level'] = 'medium';
        $risk['effort'] = 2;
        $risk['mitigation'][] = 'Verify replacement functionality';
        break;

      case 'pattern_transformation':
        $confidence = $step['confidence'] ?? 0;
        if ($confidence < 0.8) {
          $risk['level'] = 'high';
          $risk['effort'] = 4;
        }
        elseif ($confidence < 0.95) {
          $risk['level'] = 'medium';
          $risk['effort'] = 2;
        }
        $risk['mitigation'][] = 'Review automated changes';
        break;
    }

    return $risk;
  }

  /**
   * Calculates confidence score for the upgrade path.
   */
  protected function calculateConfidenceScore(array $steps) {
    if (empty($steps)) {
      return 0;
    }

    $total_confidence = 0;
    $weighted_sum = 0;
    $total_weight = 0;

    foreach ($steps as $step) {
      $weight = $this->getStepWeight($step);
      $confidence = $step['confidence'] ?? ($step['automated'] ? 0.9 : 0.6);
      
      $weighted_sum += $confidence * $weight;
      $total_weight += $weight;
    }

    return $total_weight > 0 ? $weighted_sum / $total_weight : 0;
  }

  /**
   * Gets weight for a step type.
   */
  protected function getStepWeight($step) {
    $weights = [
      'breaking_change' => 3,
      'api_update' => 2,
      'deprecation' => 1,
      'pattern_transformation' => 2,
    ];

    return $weights[$step['type']] ?? 1;
  }

  /**
   * Generates a description for a pattern transformation step.
   */
  protected function generateStepDescription($prediction) {
    $transformation = $prediction['transformation'];
    return sprintf(
      'Transform pattern: %s to %s (Confidence: %d%%)',
      $this->formatPattern($transformation['before']),
      $this->formatPattern($transformation['after']),
      $prediction['confidence'] * 100
    );
  }

  /**
   * Formats a pattern for display.
   */
  protected function formatPattern($pattern) {
    if (is_array($pattern)) {
      return json_encode($pattern, JSON_PRETTY_PRINT);
    }
    return (string) $pattern;
  }
}
