<?php

namespace Drupal\ai_upgrade_assistant\Theme;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Render\RendererInterface;

/**
 * Provides preprocessing functionality for AI Upgrade Assistant theme elements.
 */
class AiUpgradePreprocessor {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new AiUpgradePreprocessor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state,
    ThemeManagerInterface $theme_manager,
    RendererInterface $renderer
  ) {
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
    $this->state = $state;
    $this->themeManager = $theme_manager;
    $this->renderer = $renderer;
  }

  /**
   * Preprocesses upgrade analysis results.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessAnalysisResults(array &$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $theme = $this->themeManager->getActiveTheme();

    // Add theme-specific classes
    $variables['attributes']['class'][] = 'ai-upgrade-analysis';
    $variables['attributes']['class'][] = 'theme-' . $theme->getName();

    // Add responsive classes
    $variables['attributes']['class'][] = 'responsive-enabled';
    
    // Process results for display
    if (!empty($variables['results'])) {
      foreach ($variables['results'] as &$result) {
        $result['#theme'] = 'ai_upgrade_result';
        $result['#attributes']['class'][] = 'upgrade-result';
        $result['#attributes']['class'][] = $result['severity'];
        
        // Add interactive elements
        $result['#attached']['library'][] = 'ai_upgrade_assistant/results';
        
        // Add accessibility attributes
        $result['#attributes']['role'] = 'article';
        $result['#attributes']['aria-label'] = t('Upgrade analysis result');
      }
    }

    // Add theme variant
    $variant = $config->get('theme_variant') ?: 'default';
    $variables['attributes']['class'][] = 'variant-' . $variant;

    // Add animation settings
    if ($config->get('enable_animations')) {
      $variables['attributes']['class'][] = 'animated';
      $variables['#attached']['library'][] = 'ai_upgrade_assistant/animations';
    }

    // Add dark mode support
    if ($this->isDarkMode()) {
      $variables['attributes']['class'][] = 'dark-mode';
    }

    // Add performance metrics if enabled
    if ($config->get('show_performance_metrics')) {
      $variables['performance'] = $this->getPerformanceMetrics();
    }
  }

  /**
   * Preprocesses upgrade progress indicators.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessProgress(array &$variables) {
    // Add progress animation
    $variables['#attached']['library'][] = 'ai_upgrade_assistant/progress';
    
    // Calculate progress percentage
    $total = $variables['total_steps'] ?? 1;
    $current = $variables['current_step'] ?? 0;
    $variables['progress'] = round(($current / $total) * 100);

    // Add aria attributes for accessibility
    $variables['attributes']['role'] = 'progressbar';
    $variables['attributes']['aria-valuemin'] = 0;
    $variables['attributes']['aria-valuemax'] = 100;
    $variables['attributes']['aria-valuenow'] = $variables['progress'];

    // Add estimated time remaining
    if ($start_time = $this->state->get('ai_upgrade_assistant.analysis_start_time')) {
      $variables['time_remaining'] = $this->calculateTimeRemaining($start_time, $current, $total);
    }
  }

  /**
   * Preprocesses upgrade dashboard elements.
   *
   * @param array &$variables
   *   Variables to preprocess.
   */
  public function preprocessDashboard(array &$variables) {
    // Add dashboard libraries
    $variables['#attached']['library'][] = 'ai_upgrade_assistant/dashboard';
    
    // Add responsive grid classes
    $variables['attributes']['class'][] = 'dashboard-grid';
    $variables['attributes']['class'][] = 'cols-' . $this->getGridColumns();

    // Add dashboard sections
    $variables['sections'] = [
      'overview' => $this->buildOverviewSection(),
      'recent' => $this->buildRecentAnalysesSection(),
      'stats' => $this->buildStatsSection(),
      'actions' => $this->buildActionsSection(),
    ];

    // Add real-time updates if enabled
    if ($this->configFactory->get('ai_upgrade_assistant.settings')->get('enable_live_updates')) {
      $variables['#attached']['library'][] = 'ai_upgrade_assistant/live-updates';
    }
  }

  /**
   * Checks if dark mode should be enabled.
   *
   * @return bool
   *   TRUE if dark mode should be enabled.
   */
  protected function isDarkMode() {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $mode = $config->get('color_scheme');

    return $mode === 'dark' || ($mode === 'auto' && $this->isSystemDarkMode());
  }

  /**
   * Checks if system is in dark mode.
   *
   * @return bool
   *   TRUE if system is in dark mode.
   */
  protected function isSystemDarkMode() {
    // Implementation depends on how we detect system dark mode
    return FALSE;
  }

  /**
   * Gets the number of grid columns based on screen size.
   *
   * @return int
   *   Number of columns.
   */
  protected function getGridColumns() {
    $width = $this->state->get('ai_upgrade_assistant.viewport_width', 1024);
    
    if ($width >= 1200) {
      return 4;
    }
    elseif ($width >= 768) {
      return 3;
    }
    elseif ($width >= 480) {
      return 2;
    }
    
    return 1;
  }

  /**
   * Calculates estimated time remaining.
   *
   * @param int $start_time
   *   Start timestamp.
   * @param int $current
   *   Current step.
   * @param int $total
   *   Total steps.
   *
   * @return string
   *   Formatted time remaining.
   */
  protected function calculateTimeRemaining($start_time, $current, $total) {
    if ($current === 0) {
      return t('Calculating...');
    }

    $elapsed = time() - $start_time;
    $per_step = $elapsed / $current;
    $remaining = round(($total - $current) * $per_step);

    if ($remaining < 60) {
      return t('@count seconds', ['@count' => $remaining]);
    }
    
    return t('@count minutes', ['@count' => round($remaining / 60)]);
  }

  /**
   * Gets performance metrics for display.
   *
   * @return array
   *   Performance metrics render array.
   */
  protected function getPerformanceMetrics() {
    $metrics = $this->state->get('ai_upgrade_assistant.performance_metrics', []);
    
    return [
      '#theme' => 'ai_upgrade_performance_metrics',
      '#metrics' => $metrics,
      '#cache' => [
        'max-age' => 15,
        'tags' => ['ai_upgrade_assistant:performance'],
      ],
    ];
  }

}
