<?php

namespace Drupal\Tests\ai_upgrade_assistant\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the patch application functionality.
 *
 * @group ai_upgrade_assistant
 */
class PatchApplicationTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai_upgrade_assistant',
    'update',
    'system',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to administer site configuration.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'administer modules',
      'access administration pages',
      'use ai upgrade assistant',
    ]);

    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests the patch application process through the UI.
   */
  public function testPatchApplication() {
    // Visit the module analysis page.
    $this->drupalGet('admin/reports/upgrade-assistant');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('AI Upgrade Assistant');

    // Ensure the analysis form is present.
    $this->assertSession()->buttonExists('Analyze Modules');

    // Test starting the analysis.
    $this->submitForm([], 'Analyze Modules');
    $this->assertSession()->pageTextContains('Analysis in progress');

    // Wait for analysis to complete.
    $this->assertSession()->assertWaitOnAjaxRequest();

    // Check for analysis results.
    $this->assertSession()->pageTextContains('Analysis Results');
    
    // Test patch application if available.
    $page = $this->getSession()->getPage();
    $apply_button = $page->findButton('Apply Available Patches');
    
    if ($apply_button) {
      $apply_button->click();
      $this->assertSession()->assertWaitOnAjaxRequest();
      
      // Verify patch application status.
      $this->assertSession()->pageTextContains('Patch application complete');
    }
  }

  /**
   * Tests the rollback functionality.
   */
  public function testPatchRollback() {
    // Visit the module analysis page.
    $this->drupalGet('admin/reports/upgrade-assistant');
    
    // Start analysis.
    $this->submitForm([], 'Analyze Modules');
    $this->assertSession()->assertWaitOnAjaxRequest();
    
    // Apply patches if available.
    $page = $this->getSession()->getPage();
    $apply_button = $page->findButton('Apply Available Patches');
    
    if ($apply_button) {
      $apply_button->click();
      $this->assertSession()->assertWaitOnAjaxRequest();
      
      // Test rollback functionality.
      $rollback_button = $page->findButton('Rollback Changes');
      if ($rollback_button) {
        $rollback_button->click();
        $this->assertSession()->assertWaitOnAjaxRequest();
        
        // Verify rollback status.
        $this->assertSession()->pageTextContains('Changes rolled back successfully');
      }
    }
  }

  /**
   * Tests the error handling during patch application.
   */
  public function testPatchApplicationErrors() {
    // Visit the module analysis page.
    $this->drupalGet('admin/reports/upgrade-assistant');
    
    // Start analysis.
    $this->submitForm([], 'Analyze Modules');
    $this->assertSession()->assertWaitOnAjaxRequest();
    
    // Test error handling with invalid patch.
    $state = \Drupal::state();
    $state->set('ai_upgrade_assistant.test_invalid_patch', TRUE);
    
    $page = $this->getSession()->getPage();
    $apply_button = $page->findButton('Apply Available Patches');
    
    if ($apply_button) {
      $apply_button->click();
      $this->assertSession()->assertWaitOnAjaxRequest();
      
      // Verify error message.
      $this->assertSession()->pageTextContains('Error applying patch');
    }
  }

}
