<?php

namespace Drupal\Tests\ai_upgrade_assistant\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the upgrade path generation functionality.
 *
 * @group ai_upgrade_assistant
 */
class UpgradePathGenerationTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'ai_upgrade_assistant',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A test user with administrative privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create and log in our privileged user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'access upgrade analysis',
    ]);
    $this->drupalLogin($this->adminUser);

    // Create test module
    $this->createTestModule();
  }

  /**
   * Tests the upgrade path generation through the UI.
   */
  public function testUpgradePathGenerationUi() {
    // Visit the upgrade analysis page
    $this->drupalGet('admin/reports/upgrade-analysis');
    $this->assertSession()->statusCodeEquals(200);

    // Check for our test module
    $this->assertSession()->pageTextContains('Test Module');

    // Generate upgrade path
    $this->submitForm([
      'modules[test_module]' => TRUE,
      'target_version' => '9.0.0',
    ], 'Generate Upgrade Path');

    // Verify results
    $this->assertSession()->pageTextContains('Upgrade path generated successfully');
    $this->assertSession()->pageTextContains('Steps to upgrade');
    $this->assertSession()->pageTextContains('Risk assessment');

    // Verify specific upgrade steps are shown
    $this->assertSession()->pageTextContains('drupal_set_message');
    $this->assertSession()->pageTextContains('Breaking changes');
  }

  /**
   * Tests the upgrade path generation with multiple modules.
   */
  public function testMultiModuleUpgrade() {
    // Create additional test modules
    $this->createTestModuleWithDependencies();

    // Visit the upgrade analysis page
    $this->drupalGet('admin/reports/upgrade-analysis');

    // Generate upgrade path for multiple modules
    $this->submitForm([
      'modules[test_module]' => TRUE,
      'modules[test_module_dependent]' => TRUE,
      'target_version' => '9.0.0',
    ], 'Generate Upgrade Path');

    // Verify results for both modules
    $this->assertSession()->pageTextContains('Test Module');
    $this->assertSession()->pageTextContains('Test Module Dependent');
    $this->assertSession()->pageTextContains('Module dependencies');
  }

  /**
   * Tests the upgrade path export functionality.
   */
  public function testUpgradePathExport() {
    // Generate and export upgrade path
    $this->drupalGet('admin/reports/upgrade-analysis');
    $this->submitForm([
      'modules[test_module]' => TRUE,
      'target_version' => '9.0.0',
      'export_format' => 'json',
    ], 'Export Upgrade Path');

    // Verify JSON response
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseHeaderEquals('Content-Type', 'application/json');

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertIsArray($response);
    $this->assertArrayHasKey('upgrade_path', $response);
    $this->assertArrayHasKey('steps', $response['upgrade_path']);
  }

  /**
   * Creates a test module for upgrade path testing.
   */
  protected function createTestModule() {
    $module_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module';
    mkdir($module_path, 0777, TRUE);

    // Create test_module.info.yml
    file_put_contents($module_path . '/test_module.info.yml', "
name: Test Module
type: module
description: 'Test module for upgrade path generation'
core: 8.x
dependencies:
  - drupal:system (>=8.9.0)
    ");

    // Create test_module.module with deprecated code
    file_put_contents($module_path . '/test_module.module', "<?php
function test_module_form_alter(&\$form, \$form_state, \$form_id) {
  drupal_set_message('Test message');
  \Drupal::entityManager()->getStorage('node');
}
    ");
  }

  /**
   * Creates test modules with dependencies for testing.
   */
  protected function createTestModuleWithDependencies() {
    $base_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_base';
    $dependent_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_dependent';

    // Create base module
    mkdir($base_path, 0777, TRUE);
    file_put_contents($base_path . '/test_module_base.info.yml', "
name: Test Module Base
type: module
description: 'Base module for dependency testing'
core: 8.x
    ");

    // Create dependent module
    mkdir($dependent_path, 0777, TRUE);
    file_put_contents($dependent_path . '/test_module_dependent.info.yml', "
name: Test Module Dependent
type: module
description: 'Dependent module for testing'
core: 8.x
dependencies:
  - test_module_base
    ");
  }

  /**
   * {@inheritdoc}
   */
  public function tearDown(): void {
    // Clean up test modules
    $paths = [
      $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module',
      $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_base',
      $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_dependent',
    ];

    foreach ($paths as $path) {
      if (is_dir($path)) {
        $this->deleteDirectory($path);
      }
    }

    parent::tearDown();
  }

  /**
   * Recursively deletes a directory.
   */
  protected function deleteDirectory($dir) {
    if (!file_exists($dir)) {
      return;
    }

    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
      $path = $dir . '/' . $file;
      is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
    }
    return rmdir($dir);
  }

}
