<?php

namespace Drupal\Tests\ai_upgrade_assistant\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests the JavaScript functionality of the AI Upgrade Assistant.
 *
 * @group ai_upgrade_assistant
 */
class AIUpgradeAssistantJavascriptTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai_upgrade_assistant',
    'upgrade_status',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A test user with administrative privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'administer modules',
      'access administration pages',
      'administer ai upgrade assistant',
    ]);
    $this->drupalLogin($this->adminUser);

    // Configure test settings
    $this->config('ai_upgrade_assistant.settings')
      ->set('openai_api_key', 'sk-testkey123456789012345678901234567890')
      ->save();
  }

  /**
   * Tests the real-time analysis progress updates.
   */
  public function testAnalysisProgressUpdates() {
    $this->drupalGet('admin/reports/upgrade-status/ai-analysis');
    $this->assertSession()->statusCodeEquals(200);

    // Start the analysis
    $this->click('button:contains("Start Analysis")');

    // Wait for and verify progress updates
    $this->assertSession()->waitForElement('css', '.progress-bar');
    $this->assertSession()->elementExists('css', '.analysis-status');

    // Wait for completion
    $this->assertSession()->waitForText('Analysis complete');
    
    // Verify results are displayed
    $this->assertSession()->elementExists('css', '.analysis-results');
    $this->assertSession()->pageTextContains('Files Analyzed');
  }

  /**
   * Tests the interactive module selection interface.
   */
  public function testModuleSelectionInterface() {
    $this->drupalGet('admin/reports/upgrade-status/ai-analysis');

    // Test select all functionality
    $this->click('input[name="select_all"]');
    $this->assertSession()->elementAttributeContains(
      'css',
      'input[name="modules[system]"]',
      'checked',
      'checked'
    );

    // Test module filtering
    $this->getSession()->getPage()->fillField('module_filter', 'system');
    $this->assertSession()->assertWaitOnAjaxRequest();
    
    // Verify filter results
    $visible_modules = $this->getSession()->getPage()->findAll('css', '.module-item:not(.hidden)');
    foreach ($visible_modules as $module) {
      $this->assertStringContainsString('system', strtolower($module->getText()));
    }
  }

  /**
   * Tests the analysis results visualization.
   */
  public function testResultsVisualization() {
    // Generate test analysis data
    $this->generateTestAnalysisData();

    $this->drupalGet('admin/reports/upgrade-status/ai-analysis/report');

    // Test chart rendering
    $this->assertSession()->waitForElement('css', '.results-chart');
    
    // Test interactive elements
    $this->click('.severity-filter button[data-severity="medium"]');
    $this->assertSession()->assertWaitOnAjaxRequest();
    
    // Verify filtered results
    $this->assertSession()->elementTextContains(
      'css',
      '.filtered-results',
      'medium severity issues'
    );
  }

  /**
   * Tests the patch preview and application interface.
   */
  public function testPatchInterface() {
    $this->drupalGet('admin/reports/upgrade-status/ai-analysis/patches');

    // Generate a test patch
    $this->generateTestPatch();

    // Test patch preview
    $this->click('.patch-preview-button');
    $this->assertSession()->waitForElement('css', '.patch-diff');
    
    // Test patch application
    $this->click('.apply-patch-button');
    $this->assertSession()->waitForText('Patch applied successfully');
    
    // Verify patch application
    $this->assertSession()->elementTextContains(
      'css',
      '.applied-patches',
      'test_module.patch'
    );
  }

  /**
   * Generates test analysis data.
   */
  protected function generateTestAnalysisData() {
    $analysis_data = [
      'timestamp' => time(),
      'files_analyzed' => 10,
      'issues_found' => 5,
      'results' => [
        'system.module' => [
          'severity' => 'medium',
          'explanation' => 'Multiple deprecated function calls found',
          'suggestions' => [
            'Update service container usage',
            'Replace deprecated functions',
          ],
        ],
      ],
      'charts_data' => [
        'severity_distribution' => [
          'critical' => 1,
          'high' => 2,
          'medium' => 5,
          'low' => 2,
        ],
      ],
    ];
    
    \Drupal::state()->set('ai_upgrade_assistant.analysis_results', $analysis_data);
  }

  /**
   * Generates a test patch file.
   */
  protected function generateTestPatch() {
    $patch_content = <<<EOT
diff --git a/test_module.module b/test_module.module
index 1234567..89abcdef 100644
--- a/test_module.module
+++ b/test_module.module
@@ -15,7 +15,7 @@ function test_module_form_alter(&\$form, \$form_state, \$form_id) {
-  drupal_set_message('Test message');
+  \Drupal::messenger()->addStatus('Test message');
   \$language = language_default();
   return db_query('SELECT * FROM {node}');
 }
EOT;

    $patch_dir = $this->publicFilesDirectory . '/patches';
    mkdir($patch_dir, 0777, TRUE);
    file_put_contents($patch_dir . '/test_module.patch', $patch_content);
  }

}
