<?php

namespace Drupal\Tests\ai_upgrade_assistant\Kernel\Service;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Core\Extension\Extension;

/**
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\UpgradePathGenerator
 * @group ai_upgrade_assistant
 */
class UpgradePathGeneratorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'ai_upgrade_assistant',
  ];

  /**
   * The upgrade path generator service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\UpgradePathGenerator
   */
  protected $upgradePathGenerator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['system']);
    $this->upgradePathGenerator = $this->container->get('ai_upgrade_assistant.upgrade_path_generator');
  }

  /**
   * Tests the full upgrade path generation process with a real module.
   */
  public function testFullUpgradePathGeneration() {
    // Create a test module
    $module_path = $this->createTestModule();

    // Generate upgrade path
    $path = $this->upgradePathGenerator->generateUpgradePath(
      'test_module',
      '8.9.0',
      '9.0.0'
    );

    // Verify path structure
    $this->assertIsArray($path);
    $this->assertEquals('test_module', $path['module']);
    $this->assertArrayHasKey('steps', $path);
    $this->assertArrayHasKey('dependencies', $path);
    $this->assertArrayHasKey('risks', $path);

    // Verify steps contain expected transformations
    $found_deprecated_function = false;
    foreach ($path['steps'] as $step) {
      if ($step['type'] === 'pattern_transformation' &&
          strpos($step['description'], 'drupal_set_message') !== false) {
        $found_deprecated_function = true;
        break;
      }
    }
    $this->assertTrue($found_deprecated_function, 'Expected deprecated function transformation not found');

    // Verify risk assessment
    $this->assertNotEmpty($path['risks']);
    $has_high_risk = false;
    foreach ($path['risks'] as $risk) {
      if ($risk['level'] === 'high') {
        $has_high_risk = true;
        break;
      }
    }
    $this->assertTrue($has_high_risk, 'No high risk items found in assessment');

    // Verify effort estimation
    $this->assertGreaterThan(0, $path['estimated_effort']);

    // Clean up
    $this->removeTestModule();
  }

  /**
   * Tests upgrade path generation with dependencies.
   */
  public function testUpgradePathWithDependencies() {
    // Create test modules with dependencies
    $this->createTestModuleWithDependencies();

    // Generate upgrade path
    $path = $this->upgradePathGenerator->generateUpgradePath(
      'test_module_dependent',
      '8.9.0',
      '9.0.0'
    );

    // Verify dependencies are correctly analyzed
    $this->assertArrayHasKey('required', $path['dependencies']);
    $this->assertNotEmpty($path['dependencies']['required']);
    
    $found_dependency = false;
    foreach ($path['dependencies']['required'] as $dep) {
      if ($dep['name'] === 'test_module_base') {
        $found_dependency = true;
        break;
      }
    }
    $this->assertTrue($found_dependency, 'Module dependency not found');

    // Clean up
    $this->removeTestModules();
  }

  /**
   * Tests version-specific changes detection.
   */
  public function testVersionSpecificChanges() {
    $path = $this->upgradePathGenerator->generateUpgradePath(
      'system',
      '8.9.0',
      '9.0.0'
    );

    // Verify version-specific changes are detected
    $found_version_change = false;
    foreach ($path['steps'] as $step) {
      if ($step['version'] === '9.0.0' && 
          $step['type'] === 'breaking_change') {
        $found_version_change = true;
        break;
      }
    }
    $this->assertTrue($found_version_change, 'Version-specific changes not detected');
  }

  /**
   * Creates a test module for upgrade path testing.
   */
  protected function createTestModule() {
    $module_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module';
    mkdir($module_path, 0777, TRUE);

    // Create test_module.info.yml
    file_put_contents($module_path . '/test_module.info.yml', "
name: Test Module
type: module
description: 'Test module for upgrade path generation'
core: 8.x
dependencies:
  - drupal:system (>=8.9.0)
    ");

    // Create test_module.module with deprecated code
    file_put_contents($module_path . '/test_module.module', "<?php
function test_module_form_alter(&\$form, \$form_state, \$form_id) {
  drupal_set_message('Test message');
  \Drupal::entityManager()->getStorage('node');
}
    ");

    return $module_path;
  }

  /**
   * Creates test modules with dependencies for testing.
   */
  protected function createTestModuleWithDependencies() {
    $base_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_base';
    $dependent_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_dependent';

    // Create base module
    mkdir($base_path, 0777, TRUE);
    file_put_contents($base_path . '/test_module_base.info.yml', "
name: Test Module Base
type: module
description: 'Base module for dependency testing'
core: 8.x
    ");

    // Create dependent module
    mkdir($dependent_path, 0777, TRUE);
    file_put_contents($dependent_path . '/test_module_dependent.info.yml', "
name: Test Module Dependent
type: module
description: 'Dependent module for testing'
core: 8.x
dependencies:
  - test_module_base
    ");
  }

  /**
   * Removes test module.
   */
  protected function removeTestModule() {
    $module_path = $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module';
    if (is_dir($module_path)) {
      $this->deleteDirectory($module_path);
    }
  }

  /**
   * Removes test modules with dependencies.
   */
  protected function removeTestModules() {
    $paths = [
      $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_base',
      $this->container->get('kernel')->getAppRoot() . '/modules/custom/test_module_dependent',
    ];

    foreach ($paths as $path) {
      if (is_dir($path)) {
        $this->deleteDirectory($path);
      }
    }
  }

  /**
   * Recursively deletes a directory.
   */
  protected function deleteDirectory($dir) {
    if (!file_exists($dir)) {
      return;
    }

    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
      $path = $dir . '/' . $file;
      is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
    }
    return rmdir($dir);
  }

}
