<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\NodeVisitor;

use Drupal\Tests\UnitTestCase;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\NamespaceVisitor;

/**
 * Tests the namespace visitor.
 *
 * @group ai_upgrade_assistant
 */
class NamespaceVisitorTest extends UnitTestCase {

  /**
   * The PHP parser.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
    $this->traverser = new NodeTraverser();
  }

  /**
   * Tests PSR-4 compliance checking.
   */
  public function testPsr4Compliance() {
    $code = <<<'EOT'
<?php
namespace WrongNamespace\MyModule\Controller;

class MyController {
}
EOT;

    $visitor = new NamespaceVisitor('my_module', '/path/to/my_module');
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(1, $findings);
    
    $this->assertTrue($findings[0]['psr4_violation']);
    $this->assertEquals('Drupal\my_module\Controller', $findings[0]['expected_namespace']);
  }

  /**
   * Tests deprecated namespace detection.
   */
  public function testDeprecatedNamespaceDetection() {
    $code = <<<'EOT'
<?php
namespace Drupal\my_module\Controller;

use Drupal\Core\Entity\EntityNG;
use Drupal\field\Plugin\Type\Widget\WidgetBase;

class MyController {
}
EOT;

    $visitor = new NamespaceVisitor('my_module', '/path/to/my_module');
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['deprecated_uses']);
    $this->assertEquals('Drupal\Core\Entity\EntityNG', $findings[0]['deprecated_uses'][0]['name']);
  }

  /**
   * Tests namespace issue detection.
   */
  public function testNamespaceIssueDetection() {
    $code = <<<'EOT'
<?php
namespace My_Module\Controller;

class MyController {
}
EOT;

    $visitor = new NamespaceVisitor('my_module', '/path/to/my_module');
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['issues']);
    $this->assertContains('Namespace contains underscores, should use CamelCase', $findings[0]['issues']);
    $this->assertContains('Namespace should start with "Drupal\\"', $findings[0]['issues']);
  }

  /**
   * Tests test namespace validation.
   */
  public function testTestNamespaceValidation() {
    $code = <<<'EOT'
<?php
namespace Drupal\Tests\my_module\WrongTestType;

class MyTest {
}
EOT;

    $visitor = new NamespaceVisitor('my_module', '/path/to/my_module');
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['issues']);
    $this->assertContains('Test namespace should include proper test type (Unit/Kernel/Functional/FunctionalJavascript)', $findings[0]['issues']);
  }

}
