<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\NodeVisitor;

use Drupal\Tests\UnitTestCase;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\PluginVisitor;

/**
 * Tests the plugin visitor.
 *
 * @group ai_upgrade_assistant
 */
class PluginVisitorTest extends UnitTestCase {

  /**
   * The PHP parser.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
    $this->traverser = new NodeTraverser();
  }

  /**
   * Tests plugin detection.
   */
  public function testPluginDetection() {
    $code = <<<'EOT'
<?php

/**
 * @Block(
 *   id = "my_block",
 *   admin_label = @Translation("My Block")
 * )
 */
class MyBlock extends BlockBase {
  public function defaultConfiguration() {
    return [];
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    return $form;
  }
}
EOT;

    $visitor = new PluginVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(1, $findings);
    
    $this->assertEquals('plugin', $findings[0]['type']);
    $this->assertEquals('Block', $findings[0]['plugin_type']);
    $this->assertTrue($findings[0]['has_configuration']);
    $this->assertTrue($findings[0]['has_config_form']);
  }

  /**
   * Tests deprecated plugin detection.
   */
  public function testDeprecatedPluginDetection() {
    $code = <<<'EOT'
<?php

/**
 * @FieldWidget(
 *   id = "my_widget",
 *   label = @Translation("My Widget")
 * )
 */
class MyWidget extends Drupal\field\Plugin\Type\Widget\WidgetBase {
}
EOT;

    $visitor = new PluginVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertTrue($findings[0]['deprecated']);
    $this->assertEquals('Drupal\Core\Field\WidgetBase', $findings[0]['replacement']);
    $this->assertTrue($findings[0]['critical']);
  }

  /**
   * Tests Drupal 11 plugin changes detection.
   */
  public function testDrupal11PluginChangesDetection() {
    $code = <<<'EOT'
<?php

/**
 * @ViewsField(
 *   id = "my_field",
 *   title = @Translation("My Field")
 * )
 */
class MyViewsField extends FieldPluginBase {
}
EOT;

    $visitor = new PluginVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['drupal11_changes']);
    $this->assertNotEmpty($findings[0]['drupal11_changes']['changes']);
  }

}
