<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\NodeVisitor;

use Drupal\Tests\UnitTestCase;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\ServiceVisitor;

/**
 * Tests the service visitor.
 *
 * @group ai_upgrade_assistant
 */
class ServiceVisitorTest extends UnitTestCase {

  /**
   * The PHP parser.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
    $this->traverser = new NodeTraverser();
  }

  /**
   * Tests static service call detection.
   */
  public function testStaticServiceCallDetection() {
    $code = <<<'EOT'
<?php
$entity_manager = \Drupal::service('entity.manager');
$plugin_manager = \Drupal::service('plugin.manager.entity');
$url_generator = \Drupal::service('url_generator');
EOT;

    $visitor = new ServiceVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(3, $findings);
    
    $this->assertEquals('entity.manager', $findings[0]['service']);
    $this->assertTrue($findings[0]['deprecated']);
    $this->assertTrue($findings[0]['critical']);
    $this->assertEquals('entity_type.manager', $findings[0]['replacement']);
  }

  /**
   * Tests service injection detection.
   */
  public function testServiceInjectionDetection() {
    $code = <<<'EOT'
<?php
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyService implements ContainerInjectionInterface {
  protected $entityManager;
  protected $database;

  public function __construct($entity_manager, $database) {
    $this->entityManager = $entity_manager;
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.manager'),
      $container->get('database')
    );
  }
}
EOT;

    $visitor = new ServiceVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(1, $findings);
    
    $this->assertEquals('service_injection', $findings[0]['type']);
    $this->assertTrue($findings[0]['container_inject']);
    $this->assertContains('entity.manager', $findings[0]['injected_services']);
    $this->assertNotEmpty($findings[0]['deprecated_services']);
  }

  /**
   * Tests Drupal 11 service changes detection.
   */
  public function testDrupal11ServiceChangesDetection() {
    $code = <<<'EOT'
<?php
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyService implements ContainerInjectionInterface {
  protected $entityTypeManager;
  protected $database;

  public function __construct($entity_type_manager, $database) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('database')
    );
  }
}
EOT;

    $visitor = new ServiceVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['drupal11_services']);
    $found_d11_changes = false;
    foreach ($findings[0]['drupal11_services'] as $service) {
      if ($service['service'] === 'entity_type.manager') {
        $found_d11_changes = true;
        $this->assertNotEmpty($service['changes']['changes']);
        break;
      }
    }
    $this->assertTrue($found_d11_changes, 'Should detect Drupal 11 service changes');
  }

}
