/**
 * @file
 * JavaScript behaviors for the character sheet.
 */

(function ($, Drupal, once) {
  'use strict';

  /**
   * Adds interactive behaviors to the character sheet.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.aiUpgradeCharacterSheet = {
    attach: function (context, settings) {
      // Add level-up animation class when experience is gained
      once('character-sheet', '.experience-bar', context).forEach(function (element) {
        const $bar = $(element);
        const $progress = $bar.find('.experience-progress');
        const currentWidth = $progress.width();
        
        // If width changes, trigger level up animation
        const observer = new MutationObserver(function(mutations) {
          mutations.forEach(function(mutation) {
            if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
              const newWidth = $progress.width();
              if (newWidth < currentWidth) {
                // Level up occurred (progress reset)
                $bar.closest('.level-info').find('.current-level').addClass('level-up');
                setTimeout(function() {
                  $bar.closest('.level-info').find('.current-level').removeClass('level-up');
                }, 500);
              }
            }
          });
        });
        
        observer.observe($progress[0], {
          attributes: true,
          attributeFilter: ['style']
        });
      });

      // Add hover effects for achievements
      once('achievement-hover', '.achievement-item', context).forEach(function (element) {
        const $achievement = $(element);
        
        $achievement.on('mouseenter', function() {
          $(this).find('i').addClass('fa-spin');
        }).on('mouseleave', function() {
          $(this).find('i').removeClass('fa-spin');
        });
      });

      // Add click handlers for action buttons
      once('action-buttons', '.action-button', context).forEach(function (element) {
        const $button = $(element);
        
        $button.on('click', function(e) {
          e.preventDefault();
          const action = $(this).data('action');
          
          switch(action) {
            case 'quests':
              Drupal.announce(Drupal.t('Opening quest log...'));
              // TODO: Implement quest log view
              break;
              
            case 'inventory':
              Drupal.announce(Drupal.t('Opening inventory...'));
              // TODO: Implement inventory view
              break;
              
            case 'leaderboard':
              Drupal.announce(Drupal.t('Opening leaderboard...'));
              // TODO: Implement leaderboard view
              break;
          }
        });
      });

      // Add tooltips for stats
      once('stat-tooltips', '.stat-list li', context).forEach(function (element) {
        const $stat = $(element);
        const tooltipText = $stat.data('tooltip');
        
        if (tooltipText) {
          $stat.attr('title', tooltipText);
          $stat.tooltip({
            position: {
              my: 'left center',
              at: 'right+10 center'
            }
          });
        }
      });

      // Add particle effects for rare+ achievements
      once('achievement-particles', '.achievement-item.rare, .achievement-item.epic, .achievement-item.legendary', context).forEach(function (element) {
        const $achievement = $(element);
        
        $achievement.on('mouseenter', function() {
          // Add particle effect based on rarity
          if ($achievement.hasClass('legendary')) {
            addParticles(this, 'gold');
          }
          else if ($achievement.hasClass('epic')) {
            addParticles(this, 'purple');
          }
          else {
            addParticles(this, 'blue');
          }
        });
      });
    }
  };

  /**
   * Adds particle effects to an element.
   *
   * @param {HTMLElement} element
   *   The element to add particles to.
   * @param {string} color
   *   The color of the particles.
   */
  function addParticles(element, color) {
    const colors = {
      gold: '#FFD700',
      purple: '#800080',
      blue: '#0000FF'
    };
    
    const particleCount = 20;
    const $container = $(element);
    
    for (let i = 0; i < particleCount; i++) {
      const $particle = $('<div class="particle"></div>');
      const size = Math.random() * 5 + 3;
      
      $particle.css({
        position: 'absolute',
        backgroundColor: colors[color],
        width: size + 'px',
        height: size + 'px',
        borderRadius: '50%',
        pointerEvents: 'none'
      });
      
      const startX = Math.random() * $container.width();
      const startY = Math.random() * $container.height();
      
      $particle.css({
        left: startX + 'px',
        top: startY + 'px'
      });
      
      $container.append($particle);
      
      // Animate particle
      const angle = Math.random() * Math.PI * 2;
      const velocity = Math.random() * 2 + 1;
      const dx = Math.cos(angle) * velocity;
      const dy = Math.sin(angle) * velocity;
      
      $particle.animate({
        left: '+=' + (dx * 50),
        top: '+=' + (dy * 50),
        opacity: 0
      }, 1000 + Math.random() * 1000, function() {
        $(this).remove();
      });
    }
  }

})(jQuery, Drupal, once);
