<?php

namespace Drupal\ai_upgrade_assistant\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_upgrade_assistant\Service\DataAnonymizer;
use Drupal\Core\Url;

/**
 * Privacy settings form for AI Upgrade Assistant.
 */
class PrivacySettingsForm extends ConfigFormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The data anonymizer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\DataAnonymizer
   */
  protected $dataAnonymizer;

  /**
   * Constructs a PrivacySettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\ai_upgrade_assistant\Service\DataAnonymizer $data_anonymizer
   *   The data anonymizer service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    DataAnonymizer $data_anonymizer
  ) {
    parent::__construct($config_factory);
    $this->database = $database;
    $this->dataAnonymizer = $data_anonymizer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('database'),
      $container->get('ai_upgrade_assistant.data_anonymizer')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_upgrade_assistant.privacy_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_upgrade_assistant_privacy_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_upgrade_assistant.privacy_settings');

    // Add an eye-catching privacy notice.
    $form['privacy_notice'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<div class="messages messages--warning">
        <h3>🔒 Your Privacy Matters</h3>
        <p>This module can learn from the community\'s collective upgrade experiences to make everyone\'s upgrades better. 
        You\'re in complete control of what data is shared.</p>
      </div>'),
    ];

    // Data sharing settings.
    $form['sharing'] = [
      '#type' => 'details',
      '#title' => $this->t('Data Sharing Settings'),
      '#open' => TRUE,
    ];

    $form['sharing']['enable_sharing'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Pattern Sharing'),
      '#description' => $this->t('When enabled, successfully validated upgrade patterns will be anonymized and shared with the Drupal community to improve future upgrades.'),
      '#default_value' => $config->get('enable_sharing') ?? FALSE,
    ];

    // What we collect section.
    $form['sharing']['data_collection'] = [
      '#type' => 'details',
      '#title' => $this->t('What We Collect'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="enable_sharing"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['sharing']['data_collection']['collected_data'] = [
      '#theme' => 'item_list',
      '#items' => [
        '✓ ' . $this->t('Module version numbers'),
        '✓ ' . $this->t('Code patterns (anonymized)'),
        '✓ ' . $this->t('Success/failure outcomes'),
        '✓ ' . $this->t('Drupal core version'),
      ],
      '#prefix' => '<div class="messages messages--status">',
      '#suffix' => '</div>',
    ];

    $form['sharing']['data_collection']['excluded_data'] = [
      '#theme' => 'item_list',
      '#items' => [
        '❌ ' . $this->t('Variable names'),
        '❌ ' . $this->t('File paths'),
        '❌ ' . $this->t('Database content'),
        '❌ ' . $this->t('API keys or credentials'),
        '❌ ' . $this->t('Custom business logic'),
        '❌ ' . $this->t('User data'),
      ],
      '#prefix' => '<div class="messages messages--warning">',
      '#suffix' => '</div>',
    ];

    // Preview shared data.
    if ($patterns = $this->getRecentPatterns()) {
      $form['preview'] = [
        '#type' => 'details',
        '#title' => $this->t('Preview Shared Data'),
        '#description' => $this->t('See exactly what data would be shared for recent upgrade patterns:'),
        '#open' => FALSE,
        '#states' => [
          'visible' => [
            ':input[name="enable_sharing"]' => ['checked' => TRUE],
          ],
        ],
      ];

      foreach ($patterns as $pattern) {
        $preview = $this->dataAnonymizer->previewAnonymization(unserialize($pattern->pattern_data));
        
        $form['preview']['pattern_' . $pattern->id] = [
          '#type' => 'details',
          '#title' => $this->t('Pattern: @module @from → @to', [
            '@module' => $pattern->module_name,
            '@from' => $pattern->from_version,
            '@to' => $pattern->to_version,
          ]),
          '#open' => FALSE,
        ];

        $form['preview']['pattern_' . $pattern->id]['data'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['pattern-preview']],
          'original' => [
            '#type' => 'details',
            '#title' => $this->t('Original Data'),
            '#open' => FALSE,
            'content' => [
              '#markup' => '<pre>' . json_encode($preview['original'], JSON_PRETTY_PRINT) . '</pre>',
            ],
          ],
          'anonymized' => [
            '#type' => 'details',
            '#title' => $this->t('Anonymized Data (What Gets Shared)'),
            '#open' => TRUE,
            'content' => [
              '#markup' => '<pre>' . json_encode($preview['anonymized'], JSON_PRETTY_PRINT) . '</pre>',
            ],
          ],
        ];
      }
    }

    // Advanced settings.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['paranoid_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Paranoid Mode'),
      '#description' => $this->t('Run completely offline with no data sharing. The module will still work but won\'t benefit from community learning.'),
      '#default_value' => $config->get('paranoid_mode') ?? FALSE,
    ];

    $form['advanced']['retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Data Retention Period (Days)'),
      '#description' => $this->t('Number of days to keep local patterns before automatic deletion. Set to 0 to keep indefinitely.'),
      '#default_value' => $config->get('retention_days') ?? 30,
      '#min' => 0,
      '#max' => 365,
    ];

    $form['advanced']['sync_frequency'] = [
      '#type' => 'select',
      '#title' => $this->t('Pattern Sync Frequency'),
      '#description' => $this->t('How often to sync successful patterns with the community.'),
      '#options' => [
        'immediate' => $this->t('Immediately after validation'),
        'daily' => $this->t('Daily'),
        'weekly' => $this->t('Weekly'),
        'manual' => $this->t('Manual only'),
      ],
      '#default_value' => $config->get('sync_frequency') ?? 'daily',
      '#states' => [
        'visible' => [
          ':input[name="enable_sharing"]' => ['checked' => TRUE],
          ':input[name="paranoid_mode"]' => ['checked' => FALSE],
        ],
      ],
    ];

    // Add custom styling.
    $form['#attached']['library'][] = 'ai_upgrade_assistant/privacy_settings';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('enable_sharing') && $form_state->getValue('paranoid_mode')) {
      $form_state->setError($form['advanced']['paranoid_mode'], $this->t('You cannot enable both data sharing and paranoid mode.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_upgrade_assistant.privacy_settings');
    
    $config
      ->set('enable_sharing', $form_state->getValue('enable_sharing'))
      ->set('paranoid_mode', $form_state->getValue('paranoid_mode'))
      ->set('retention_days', $form_state->getValue('retention_days'))
      ->set('sync_frequency', $form_state->getValue('sync_frequency'))
      ->save();

    // If paranoid mode is enabled, clear any pending patterns.
    if ($form_state->getValue('paranoid_mode')) {
      $this->database->update('ai_upgrade_patterns')
        ->fields(['sync_status' => 'local_only'])
        ->condition('sync_status', 'pending')
        ->execute();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets recent upgrade patterns for preview.
   *
   * @return array
   *   Array of recent patterns.
   */
  protected function getRecentPatterns() {
    return $this->database->select('ai_upgrade_patterns', 'p')
      ->fields('p')
      ->condition('sync_status', 'pending')
      ->orderBy('created', 'DESC')
      ->range(0, 3)
      ->execute()
      ->fetchAll();
  }

}
