<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for interacting with HuggingFace's Hub.
 */
class HuggingFaceService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The HuggingFace API base URL.
   *
   * @var string
   */
  protected const API_BASE_URL = 'https://huggingface.co/api';

  /**
   * The dataset repository name.
   *
   * @var string
   */
  protected const DATASET_REPO = 'drupal/ai-upgrade-patterns';

  /**
   * Constructs a new HuggingFaceService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
    $this->state = $state;
  }

  /**
   * Gets the HuggingFace API token from configuration.
   *
   * @return string|null
   *   The API token if configured, NULL otherwise.
   */
  protected function getApiToken() {
    return $this->configFactory->get('ai_upgrade_assistant.settings')->get('huggingface_token');
  }

  /**
   * Shares an upgrade pattern to the public dataset.
   *
   * @param array $pattern
   *   The upgrade pattern to share.
   *
   * @return bool
   *   TRUE if the pattern was shared successfully, FALSE otherwise.
   */
  public function sharePattern(array $pattern) {
    try {
      $token = $this->getApiToken();
      if (empty($token)) {
        $this->loggerFactory->warning('Cannot share pattern: HuggingFace API token not configured.');
        return FALSE;
      }

      $response = $this->httpClient->request('POST', self::API_BASE_URL . '/datasets/' . self::DATASET_REPO . '/push', [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
          'Content-Type' => 'application/json',
        ],
        'json' => [
          'commit_message' => 'Add new upgrade pattern',
          'content' => $pattern,
        ],
      ]);

      if ($response->getStatusCode() === 200) {
        $this->loggerFactory->info('Successfully shared upgrade pattern to HuggingFace dataset.');
        return TRUE;
      }
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->error('Error sharing pattern to HuggingFace: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * Gets statistics about the public dataset.
   *
   * @return array
   *   Array containing dataset statistics:
   *   - total_patterns: Total number of patterns
   *   - contributors: Number of unique contributors
   *   - last_update: Timestamp of last update
   *   - success_rate: Pattern success rate percentage
   */
  public function getDatasetStats() {
    try {
      $response = $this->httpClient->request('GET', self::API_BASE_URL . '/datasets/' . self::DATASET_REPO . '/stats');
      
      if ($response->getStatusCode() === 200) {
        $stats = json_decode($response->getBody(), TRUE);
        
        // Cache the stats in state for 1 hour
        $this->state->set('ai_upgrade_assistant.dataset_stats', [
          'data' => $stats,
          'timestamp' => \Drupal::time()->getRequestTime(),
        ]);
        
        return $stats;
      }
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->error('Error fetching HuggingFace dataset stats: @error', ['@error' => $e->getMessage()]);
      
      // Return cached stats if available
      $cached = $this->state->get('ai_upgrade_assistant.dataset_stats');
      if ($cached && ($cached['timestamp'] > (\Drupal::time()->getRequestTime() - 3600))) {
        return $cached['data'];
      }
    }

    return [];
  }

  /**
   * Gets a preview of patterns that will be shared.
   *
   * @param int $limit
   *   Maximum number of patterns to return.
   *
   * @return array
   *   Array of patterns ready for sharing.
   */
  public function getPatternPreview($limit = 5) {
    try {
      $response = $this->httpClient->request('GET', self::API_BASE_URL . '/datasets/' . self::DATASET_REPO . '/preview', [
        'query' => ['limit' => $limit],
      ]);

      if ($response->getStatusCode() === 200) {
        return json_decode($response->getBody(), TRUE);
      }
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->error('Error fetching pattern preview: @error', ['@error' => $e->getMessage()]);
    }

    return [];
  }

  /**
   * Downloads the latest patterns from the dataset.
   *
   * @return array
   *   Array of downloaded patterns.
   */
  public function downloadPatterns() {
    try {
      $response = $this->httpClient->request('GET', self::API_BASE_URL . '/datasets/' . self::DATASET_REPO . '/patterns');

      if ($response->getStatusCode() === 200) {
        $patterns = json_decode($response->getBody(), TRUE);
        
        // Cache the patterns
        $this->state->set('ai_upgrade_assistant.patterns', [
          'data' => $patterns,
          'timestamp' => \Drupal::time()->getRequestTime(),
        ]);
        
        return $patterns;
      }
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->error('Error downloading patterns: @error', ['@error' => $e->getMessage()]);
      
      // Return cached patterns if available
      $cached = $this->state->get('ai_upgrade_assistant.patterns');
      if ($cached && ($cached['timestamp'] > (\Drupal::time()->getRequestTime() - 86400))) {
        return $cached['data'];
      }
    }

    return [];
  }

  /**
   * Validates a HuggingFace API token.
   *
   * @param string $token
   *   The token to validate.
   *
   * @return bool
   *   TRUE if the token is valid, FALSE otherwise.
   */
  public function validateToken($token) {
    try {
      $response = $this->httpClient->request('GET', self::API_BASE_URL . '/whoami', [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
        ],
      ]);

      return $response->getStatusCode() === 200;
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->error('Error validating HuggingFace token: @error', ['@error' => $e->getMessage()]);
      return FALSE;
    }
  }

}
