<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for analyzing modules and their upgrade status.
 */
class ModuleAnalyzerService {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a ModuleAnalyzerService object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    StateInterface $state
  ) {
    $this->moduleHandler = $module_handler;
    $this->state = $state;
  }

  /**
   * Gets the status of all modules.
   *
   * @return array
   *   An array of module statuses grouped by category.
   */
  public function getModuleStatuses() {
    $modules = $this->moduleHandler->getModuleList();
    $statuses = [];

    foreach ($modules as $name => $module) {
      $status = $this->analyzeModule($name);
      $group = $this->determineModuleGroup($status);
      
      $statuses[$group][] = [
        '#type' => 'container',
        'name' => [
          '#type' => 'markup',
          '#markup' => $name,
        ],
        'status' => [
          '#type' => 'markup',
          '#markup' => $status['status'],
        ],
        'version' => [
          '#type' => 'markup',
          '#markup' => $status['version'],
        ],
        'progress' => [
          '#type' => 'markup',
          '#markup' => $status['progress'] ?? 0,
        ],
      ];
    }

    return $statuses;
  }

  /**
   * Analyzes a specific module.
   *
   * @param string $module_name
   *   The name of the module to analyze.
   *
   * @return array
   *   Analysis results for the module.
   */
  protected function analyzeModule($module_name) {
    // Get stored analysis data or perform new analysis.
    $analysis = $this->state->get("ai_upgrade_assistant.module_analysis.{$module_name}", []);
    
    if (empty($analysis)) {
      // Default status for new modules.
      $analysis = [
        'status' => 'needs_analysis',
        'version' => $this->getModuleVersion($module_name),
        'progress' => 0,
      ];
    }

    return $analysis;
  }

  /**
   * Gets the version of a module.
   *
   * @param string $module_name
   *   The name of the module.
   *
   * @return string
   *   The module version.
   */
  protected function getModuleVersion($module_name) {
    $module_info = $this->moduleHandler->getModule($module_name)->info;
    return $module_info['version'] ?? '8.x-1.x';
  }

  /**
   * Determines the group for a module based on its status.
   *
   * @param array $status
   *   The module status array.
   *
   * @return string
   *   The group name.
   */
  protected function determineModuleGroup($status) {
    switch ($status['status']) {
      case 'compatible':
        return 'Compatible';
      case 'needs_update':
        return 'Needs_Update';
      case 'in_progress':
        return 'In_Progress';
      case 'error':
        return 'Error';
      default:
        return 'Unknown';
    }
  }

  /**
   * Calculates the overall progress of module updates.
   *
   * @return array
   *   An array containing progress statistics.
   */
  public function calculateOverallProgress() {
    $modules = $this->moduleHandler->getModuleList();
    $total = count($modules);
    $completed = 0;

    foreach ($modules as $name => $module) {
      $status = $this->analyzeModule($name);
      if ($status['status'] === 'compatible') {
        $completed++;
      }
    }

    return [
      'total' => [
        '#type' => 'markup',
        '#markup' => $total,
      ],
      'completed' => [
        '#type' => 'markup',
        '#markup' => $completed,
      ],
      'percentage' => [
        '#type' => 'markup',
        '#markup' => ($total > 0) ? round(($completed / $total) * 100) : 0,
      ],
    ];
  }

  /**
   * Gets the terminal output.
   *
   * @return array
   *   An array of terminal output lines.
   */
  public function getTerminalOutput() {
    return $this->state->get('ai_upgrade_assistant.terminal_output', []);
  }

}
