<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for interacting with OpenAI API.
 */
class OpenAIService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Constructs a new OpenAIService.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->state = $state;
    $this->cache = $cache;
  }

  /**
   * Generates a completion using OpenAI API.
   *
   * @param string $prompt
   *   The prompt to generate completion for.
   *
   * @return string
   *   The generated completion.
   *
   * @throws \Exception
   *   If the API request fails.
   */
  public function generateCompletion($prompt) {
    $cid = 'openai_completion:' . md5($prompt);
    if ($cached = $this->cache->get($cid)) {
      return $cached->data;
    }

    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $api_key = $config->get('openai_api_key');
    
    if (!$api_key) {
      throw new \Exception('OpenAI API key not configured');
    }

    $model_settings = $config->get('model_settings');
    $data = [
      'model' => $model_settings['model'] ?? 'gpt-4',
      'messages' => [
        [
          'role' => 'system',
          'content' => 'You are an expert Drupal developer analyzing code changes for upgrade patterns.',
        ],
        [
          'role' => 'user',
          'content' => $prompt,
        ],
      ],
      'temperature' => $model_settings['temperature'] ?? 0.7,
      'max_tokens' => $model_settings['max_tokens'] ?? 1000,
    ];

    try {
      $response = $this->httpClient->request('POST', 'https://api.openai.com/v1/chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => $data,
      ]);

      $result = json_decode($response->getBody(), TRUE);

      if (!empty($result['choices'][0]['message']['content'])) {
        $completion = $result['choices'][0]['message']['content'];
        $this->cache->set($cid, $completion, time() + 3600);
        $this->updateRateLimitStatus($response->getHeaders());
        return $completion;
      }

      throw new \Exception('No completion generated');
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')
        ->error('OpenAI API request failed: @message', ['@message' => $e->getMessage()]);
      throw new \Exception('OpenAI API request failed: ' . $e->getMessage());
    }
  }

  /**
   * Analyzes code for potential upgrade issues.
   *
   * @param string $code
   *   The code to analyze.
   * @param array $context
   *   Additional context for the analysis.
   *
   * @return array
   *   Analysis results containing:
   *   - issues: Array of potential issues found
   *   - suggestions: Array of suggested fixes
   *   - compatibility: Overall compatibility assessment
   *
   * @throws \Exception
   *   If the API request fails.
   */
  public function analyzeCode($code, array $context = []) {
    $cid = 'openai_code_analysis:' . md5($code . serialize($context));
    if ($cached = $this->cache->get($cid)) {
      return $cached->data;
    }

    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $api_key = $config->get('openai_api_key');
    
    if (!$api_key) {
      throw new \Exception('OpenAI API key is not configured.');
    }

    try {
      $prompt = $this->buildCodeAnalysisPrompt($code, $context);
      
      $response = $this->httpClient->post('https://api.openai.com/v1/chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => [
          'model' => 'gpt-4',
          'messages' => [
            [
              'role' => 'system',
              'content' => 'You are a Drupal code analyzer specializing in identifying compatibility issues and suggesting fixes for module upgrades.',
            ],
            [
              'role' => 'user',
              'content' => $prompt,
            ],
          ],
          'temperature' => 0.3,
          'max_tokens' => 1000,
        ],
      ]);

      $result = json_decode((string) $response->getBody(), TRUE);
      
      if (empty($result['choices'][0]['message']['content'])) {
        throw new \Exception('Invalid response from OpenAI API.');
      }

      $analysis = $this->parseAnalysisResponse($result['choices'][0]['message']['content']);
      
      // Cache for 24 hours
      $this->cache->set($cid, $analysis, time() + 86400);
      
      return $analysis;
    }
    catch (GuzzleException $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'OpenAI API request failed: @error',
        ['@error' => $e->getMessage()]
      );
      throw new \Exception('Failed to analyze code: ' . $e->getMessage());
    }
  }

  /**
   * Builds the prompt for code analysis.
   *
   * @param string $code
   *   The code to analyze.
   * @param array $context
   *   Additional context for analysis.
   *
   * @return string
   *   The formatted prompt.
   */
  protected function buildCodeAnalysisPrompt($code, array $context) {
    $prompt = "Analyze the following Drupal code for compatibility issues:\n\n";
    $prompt .= "Code:\n```php\n$code\n```\n\n";
    
    if (!empty($context['from_version'])) {
      $prompt .= "Current Drupal version: {$context['from_version']}\n";
    }
    if (!empty($context['to_version'])) {
      $prompt .= "Target Drupal version: {$context['to_version']}\n";
    }
    
    $prompt .= "\nPlease identify:\n";
    $prompt .= "1. Deprecated code usage\n";
    $prompt .= "2. Incompatible API changes\n";
    $prompt .= "3. Potential security issues\n";
    $prompt .= "4. Suggested fixes with code examples\n";
    
    return $prompt;
  }

  /**
   * Parses the OpenAI response into structured analysis data.
   *
   * @param string $response
   *   The raw response from OpenAI.
   *
   * @return array
   *   Structured analysis data.
   */
  protected function parseAnalysisResponse($response) {
    // Basic structure for the analysis
    $analysis = [
      'issues' => [],
      'suggestions' => [],
      'compatibility' => 'unknown',
    ];

    // Extract issues (assumed to be listed with numbers or bullet points)
    if (preg_match_all('/(?:^|\n)(?:\d+\.|-)(.+?)(?=(?:\n\d+\.|\n-|\n\n|$))/s', $response, $matches)) {
      foreach ($matches[1] as $issue) {
        $analysis['issues'][] = trim($issue);
      }
    }

    // Extract code suggestions (anything between ```php and ```)
    if (preg_match_all('/```php\n(.+?)\n```/s', $response, $matches)) {
      foreach ($matches[1] as $suggestion) {
        $analysis['suggestions'][] = trim($suggestion);
      }
    }

    // Determine overall compatibility
    if (stripos($response, 'compatible') !== false) {
      if (stripos($response, 'not compatible') !== false || stripos($response, 'incompatible') !== false) {
        $analysis['compatibility'] = 'incompatible';
      }
      else {
        $analysis['compatibility'] = 'compatible';
      }
    }

    return $analysis;
  }

  /**
   * Gets the current rate limit status.
   *
   * @return array
   *   The rate limit status.
   */
  public function getRateLimitStatus() {
    return $this->state->get('ai_upgrade_assistant.openai_rate_limit', [
      'requests_remaining' => 0,
      'requests_reset_at' => 0,
    ]);
  }

  /**
   * Updates the rate limit status.
   *
   * @param array $headers
   *   The response headers from OpenAI API.
   */
  protected function updateRateLimitStatus($headers) {
    $status = [
      'requests_remaining' => $headers['x-ratelimit-remaining-requests'][0] ?? 0,
      'requests_reset_at' => $headers['x-ratelimit-reset-requests'][0] ?? 0,
    ];
    $this->state->set('ai_upgrade_assistant.openai_rate_limit', $status);
  }

  /**
   * Get current rate limits for OpenAI API.
   *
   * @return array
   *   Array containing rate limit information:
   *   - requests_remaining: Number of requests remaining in the current window
   *   - tokens_remaining: Number of tokens remaining in the current window
   *   - reset_time: Timestamp when the rate limits will reset
   */
  public function getRateLimits() {
    $state = \Drupal::state();
    
    return [
      'requests_remaining' => $state->get('ai_upgrade_assistant.openai_requests_remaining', 100),
      'tokens_remaining' => $state->get('ai_upgrade_assistant.openai_tokens_remaining', 100000),
      'reset_time' => $state->get('ai_upgrade_assistant.openai_reset_time', strtotime('+1 hour')),
    ];
  }

}
