/**
 * @file
 * JavaScript behaviors for the AI Upgrade Assistant forum.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  /**
   * Forum behaviors.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.aiUpgradeAssistantForum = {
    attach: function (context, settings) {
      // Initialize file upload previews
      $('.field--type-file', context).once('ai-upgrade-assistant-file-preview').each(function () {
        var $field = $(this);
        var $input = $field.find('input[type="file"]');
        var $preview = $('<div class="file-preview"></div>').insertAfter($input);

        $input.on('change', function (e) {
          $preview.empty();
          var files = e.target.files;

          for (var i = 0; i < files.length; i++) {
            var file = files[i];
            var $fileInfo = $('<div class="file-info"></div>');
            $fileInfo.append('<span class="file-name">' + file.name + '</span>');
            $fileInfo.append('<span class="file-size">(' + Drupal.formatSize(file.size) + ')</span>');
            $preview.append($fileInfo);
          }
        });
      });

      // Initialize tag input enhancement
      $('input[name="tags"]', context).once('ai-upgrade-assistant-tags').each(function () {
        var $input = $(this);
        var $wrapper = $('<div class="tags-wrapper"></div>').insertAfter($input);
        var $tagList = $('<div class="tag-list"></div>').appendTo($wrapper);
        
        function updateTags() {
          var tags = $input.val().split(',').map(function (tag) {
            return tag.trim();
          }).filter(function (tag) {
            return tag.length > 0;
          });

          $tagList.empty();
          tags.forEach(function (tag) {
            var $tag = $('<span class="tag"></span>').text(tag);
            var $remove = $('<span class="remove-tag">×</span>').appendTo($tag);
            
            $remove.on('click', function () {
              var newTags = tags.filter(function (t) {
                return t !== tag;
              });
              $input.val(newTags.join(', '));
              updateTags();
            });

            $tagList.append($tag);
          });
        }

        $input.on('change keyup', updateTags);
        updateTags();
      });

      // Initialize AJAX loading for forum topics
      $('.forum-topics', context).once('ai-upgrade-assistant-ajax-load').each(function () {
        var $container = $(this);
        var page = 0;
        var loading = false;

        // Load more topics when reaching bottom of page
        $(window).on('scroll', function () {
          if (loading) return;

          var threshold = 200;
          var bottomPosition = $(document).height() - $(window).height() - threshold;

          if ($(window).scrollTop() >= bottomPosition) {
            loading = true;
            page++;

            $.ajax({
              url: drupalSettings.path.baseUrl + 'admin/reports/ai-upgrade-assistant/forum/api/topics',
              data: {
                page: page,
                tag: $('input[name="tag"]').val()
              },
              success: function (response) {
                if (response.topics.length > 0) {
                  response.topics.forEach(function (topic) {
                    var $topic = $(Drupal.theme('forumTopic', topic));
                    $container.append($topic);
                    Drupal.attachBehaviors($topic[0]);
                  });
                  loading = false;
                }
              }
            });
          }
        });
      });
    }
  };

  /**
   * Theme function for forum topics.
   *
   * @param {Object} topic
   *   The topic object.
   *
   * @return {string}
   *   The HTML markup for the topic.
   */
  Drupal.theme.forumTopic = function (topic) {
    var tags = '';
    if (topic.tags) {
      var tagList = JSON.parse(topic.tags);
      tags = '<div class="topic-tags">' + tagList.map(function (tag) {
        return '<span class="tag">' + Drupal.checkPlain(tag) + '</span>';
      }).join('') + '</div>';
    }

    return '<div class="forum-topic">' +
      '<h3 class="topic-title">' +
        '<a href="' + drupalSettings.path.baseUrl + 'admin/reports/ai-upgrade-assistant/forum/' + topic.id + '">' +
          Drupal.checkPlain(topic.title) +
        '</a>' +
      '</h3>' +
      '<div class="topic-meta">' +
        '<span class="topic-author">' + Drupal.t('By') + ' ' + Drupal.checkPlain(topic.author) + '</span>' +
        '<span class="topic-date">' + Drupal.formatDate(topic.created * 1000) + '</span>' +
        tags +
      '</div>' +
      '<div class="topic-preview">' +
        Drupal.checkPlain(topic.content.length > 200 ? topic.content.substr(0, 200) + '...' : topic.content) +
      '</div>' +
      '<div class="topic-stats">' +
        '<span class="replies">' + topic.reply_count + ' ' + Drupal.t('replies') + '</span>' +
        '<span class="views">' + topic.view_count + ' ' + Drupal.t('views') + '</span>' +
      '</div>' +
    '</div>';
  };

})(jQuery, Drupal, drupalSettings);
