<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Service for handling batch processing of module analysis.
 */
class BatchAnalyzer {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * The project analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
   */
  protected $projectAnalyzer;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The analysis tracker service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AnalysisTracker
   */
  protected $analysisTracker;

  /**
   * The patch searcher service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchSearcher
   */
  protected $patchSearcher;

  /**
   * The cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The batch size for processing.
   *
   * @var int
   */
  protected $batchSize = 5;

  /**
   * Constructs a new BatchAnalyzer object.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer $project_analyzer
   *   The project analyzer service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\ai_upgrade_assistant\Service\AnalysisTracker $analysis_tracker
   *   The analysis tracker service.
   * @param \Drupal\ai_upgrade_assistant\Service\PatchSearcher $patch_searcher
   *   The patch searcher service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   */
  public function __construct(
    ProjectAnalyzer $project_analyzer,
    StateInterface $state,
    ModuleHandlerInterface $module_handler,
    AnalysisTracker $analysis_tracker,
    PatchSearcher $patch_searcher,
    CacheBackendInterface $cache
  ) {
    $this->projectAnalyzer = $project_analyzer;
    $this->state = $state;
    $this->moduleHandler = $module_handler;
    $this->analysisTracker = $analysis_tracker;
    $this->patchSearcher = $patch_searcher;
    $this->cache = $cache;
  }

  /**
   * Creates a batch for analyzing modules.
   *
   * @param array $modules
   *   Array of module names to analyze.
   *
   * @return array
   *   Batch definition array.
   */
  public function createBatch(array $modules) {
    // Clear previous analysis results.
    $this->state->delete('ai_upgrade_assistant.last_analysis_results');
    $this->cache->deleteMultiple(['ai_upgrade_assistant_analysis']);

    // Group modules into chunks for better performance.
    $chunks = array_chunk($modules, $this->batchSize);
    $operations = [];

    foreach ($chunks as $chunk) {
      $operations[] = [
        [$this, 'analyzeBatchChunk'],
        [$chunk],
      ];
    }

    return [
      'title' => $this->t('Analyzing modules'),
      'operations' => $operations,
      'finished' => [$this, 'batchFinished'],
      'progressive' => TRUE,
      'init_message' => $this->t('Starting module analysis...'),
      'progress_message' => $this->t('Analyzed @current out of @total module groups.'),
      'error_message' => $this->t('An error occurred during the analysis.'),
    ];
  }

  /**
   * Process a batch chunk of modules.
   *
   * @param array $modules
   *   Array of module names in this chunk.
   * @param array $context
   *   Batch context array.
   */
  public function analyzeBatchChunk(array $modules, array &$context) {
    if (!isset($context['results'])) {
      $context['results'] = [];
    }

    $cache_key = 'ai_upgrade_assistant_analysis';
    $cached_data = $this->cache->get($cache_key);
    $analysis_results = $cached_data ? $cached_data->data : [];

    foreach ($modules as $module) {
      try {
        // Check cache first.
        $module_cache_key = $cache_key . ':' . $module;
        $cached_module = $this->cache->get($module_cache_key);

        if ($cached_module) {
          $result = $cached_module->data;
        }
        else {
          $result = $this->projectAnalyzer->analyzeModule($module);
          // Cache the result for 1 hour.
          $this->cache->set($module_cache_key, $result, time() + 3600);
        }

        // Track the analysis.
        $this->analysisTracker->trackModuleAnalysis($module, $result);

        // Search for available patches.
        if (!empty($result['issues'])) {
          $patches = $this->patchSearcher->searchPatches($module);
          if (!empty($patches)) {
            $result['available_patches'] = $patches;
          }
        }

        $context['results'][$module] = $result;
        $analysis_results[$module] = $result;

        $context['message'] = $this->t('Analyzed module: @module', ['@module' => $module]);
      }
      catch (\Exception $e) {
        $error_result = [
          'status' => 'error',
          'message' => $e->getMessage(),
        ];
        $context['results'][$module] = $error_result;
        $analysis_results[$module] = $error_result;
        $this->analysisTracker->trackModuleAnalysis($module, $error_result);
      }
    }

    // Update the cache with new results.
    $this->cache->set($cache_key, $analysis_results);
  }

  /**
   * Batch finished callback.
   */
  public function batchFinished($success, array $results, array $operations) {
    if ($success) {
      // Store the final results.
      $this->state->set('ai_upgrade_assistant.last_analysis_results', $results);

      // Calculate and store statistics.
      $stats = $this->calculateAnalysisStats($results);
      $this->state->set('ai_upgrade_assistant.analysis_stats', $stats);

      // Clear old cached data.
      $this->cleanupCache();
    }
  }

  /**
   * Calculate statistics from analysis results.
   *
   * @param array $results
   *   Analysis results array.
   *
   * @return array
   *   Statistics array.
   */
  protected function calculateAnalysisStats(array $results) {
    $stats = [
      'total' => count($results),
      'compatible' => 0,
      'needs_update' => 0,
      'error' => 0,
      'with_patches' => 0,
    ];

    foreach ($results as $result) {
      if ($result['status'] === 'compatible') {
        $stats['compatible']++;
      }
      elseif ($result['status'] === 'needs_update') {
        $stats['needs_update']++;
      }
      elseif ($result['status'] === 'error') {
        $stats['error']++;
      }

      if (!empty($result['available_patches'])) {
        $stats['with_patches']++;
      }
    }

    return $stats;
  }

  /**
   * Clean up old cached data.
   */
  protected function cleanupCache() {
    // Clear analysis cache tags.
    $this->cache->deleteAll();
    $this->moduleHandler->invokeAll('ai_upgrade_assistant_analysis_complete');
  }

  /**
   * Creates a batch for analyzing a single module.
   *
   * @param string $module
   *   The machine name of the module to analyze.
   *
   * @return array
   *   A batch array ready for batch_set().
   */
  public function createModuleAnalysisBatch($module) {
    $operations = [];
    $module_info = $this->moduleHandler->getModule($module);

    // Initialize analysis
    $this->analysisTracker->initializeAnalysis($module);

    // Add operations for code analysis
    $operations[] = [
      [$this, 'analyzeModuleCode'],
      [$module, $module_info->getPath()],
    ];

    // Add operations for dependency analysis
    $operations[] = [
      [$this, 'analyzeModuleDependencies'],
      [$module],
    ];

    // Add operations for patch search
    $operations[] = [
      [$this, 'searchModulePatches'],
      [$module],
    ];

    // Add operations for security analysis
    $operations[] = [
      [$this, 'analyzeModuleSecurity'],
      [$module],
    ];

    return [
      'operations' => $operations,
      'finished' => [$this, 'finishModuleAnalysis'],
      'title' => $this->t('Analyzing @module module', ['@module' => $module]),
      'init_message' => $this->t('Starting analysis of @module module...', ['@module' => $module]),
      'progress_message' => $this->t('Analyzing @module... @current out of @total steps completed.', ['@module' => $module]),
      'error_message' => $this->t('Analysis of @module has encountered an error.', ['@module' => $module]),
    ];
  }

  /**
   * Batch operation callback to analyze module code.
   *
   * @param string $module
   *   The module name.
   * @param string $path
   *   The module path.
   * @param array $context
   *   The batch context.
   */
  public function analyzeModuleCode($module, $path, &$context) {
    // Ensure services are properly loaded after serialization
    $this->ensureServices();
    
    $analysis = $this->projectAnalyzer->analyzeCode($path);
    $this->analysisTracker->updateAnalysis($module, $analysis, 'code_analysis');
    $context['message'] = $this->t('Analyzed code for @module', ['@module' => $module]);
    $context['results']['code_analysis'] = $analysis;
  }

  /**
   * Batch operation callback to analyze module dependencies.
   *
   * @param string $module
   *   The module name.
   * @param array $context
   *   The batch context.
   */
  public function analyzeModuleDependencies($module, &$context) {
    // Ensure services are properly loaded after serialization
    $this->ensureServices();
    
    $dependencies = $this->projectAnalyzer->analyzeDependencies($module);
    $this->analysisTracker->updateAnalysis($module, $dependencies, 'dependency_analysis');
    $context['message'] = $this->t('Analyzed dependencies for @module', ['@module' => $module]);
    $context['results']['dependency_analysis'] = $dependencies;
  }

  /**
   * Batch operation callback to search for module patches.
   *
   * @param string $module
   *   The module name.
   * @param array $context
   *   The batch context.
   */
  public function searchModulePatches($module, &$context) {
    // Ensure services are properly loaded after serialization
    $this->ensureServices();
    
    $patches = $this->patchSearcher->findPatches($module);
    $this->analysisTracker->updateAnalysis($module, $patches, 'patches');
    $context['message'] = $this->t('Found patches for @module', ['@module' => $module]);
    $context['results']['patches'] = $patches;
  }

  /**
   * Batch operation callback to analyze module security.
   *
   * @param string $module
   *   The module name.
   * @param array $context
   *   The batch context.
   */
  public function analyzeModuleSecurity($module, &$context) {
    // Ensure services are properly loaded after serialization
    $this->ensureServices();
    
    $security = $this->projectAnalyzer->analyzeSecurityIssues($module);
    $this->analysisTracker->updateAnalysis($module, $security, 'security_analysis');
    $context['message'] = $this->t('Analyzed security for @module', ['@module' => $module]);
    $context['results']['security_analysis'] = $security;
  }

  /**
   * Batch finish callback.
   *
   * @param bool $success
   *   Whether the batch succeeded.
   * @param array $results
   *   The batch results.
   * @param array $operations
   *   The operations that remained unprocessed.
   */
  public function finishModuleAnalysis($success, $results, $operations) {
    // Ensure services are properly loaded after serialization
    $this->ensureServices();

    if ($success) {
      \Drupal::messenger()->addStatus($this->t('Module analysis completed successfully.'));
      // Cache the combined results
      $this->cache->set('ai_upgrade_assistant.analysis_results', $results);
    }
    else {
      \Drupal::messenger()->addError($this->t('Module analysis encountered errors.'));
    }
  }

  /**
   * Ensures all required services are loaded after deserialization.
   */
  protected function ensureServices() {
    if (!$this->projectAnalyzer) {
      $this->projectAnalyzer = \Drupal::service('ai_upgrade_assistant.project_analyzer');
    }
    if (!$this->analysisTracker) {
      $this->analysisTracker = \Drupal::service('ai_upgrade_assistant.analysis_tracker');
    }
    if (!$this->patchSearcher) {
      $this->patchSearcher = \Drupal::service('ai_upgrade_assistant.patch_searcher');
    }
    if (!$this->state) {
      $this->state = \Drupal::state();
    }
    if (!$this->moduleHandler) {
      $this->moduleHandler = \Drupal::moduleHandler();
    }
    if (!$this->cache) {
      $this->cache = \Drupal::cache();
    }
  }

  /**
   * Sets the batch size.
   *
   * @param int $size
   *   The batch size.
   */
  public function setBatchSize($size) {
    $this->batchSize = max(1, (int) $size);
  }

  /**
   * Creates a batch for analyzing modules.
   *
   * @return array
   *   The batch definition.
   */
  public function createAnalysisBatch() {
    $modules = $this->moduleHandler->getModuleList();
    $operations = [];

    foreach ($modules as $name => $extension) {
      // Skip core modules and non-custom modules
      if ($extension->origin === 'core' || !$this->moduleHandler->moduleExists($name)) {
        continue;
      }

      $operations[] = [
        [$this, 'processModuleAnalysis'],
        [$name],
      ];
    }

    return [
      'operations' => $operations,
      'finished' => [$this, 'finishAnalysisBatch'],
      'title' => $this->t('Analyzing modules for upgrade'),
      'init_message' => $this->t('Starting module analysis...'),
      'progress_message' => $this->t('Analyzed @current out of @total modules.'),
      'error_message' => $this->t('An error occurred during module analysis.'),
    ];
  }

  /**
   * Process a single module in the analysis batch.
   *
   * @param string $module_name
   *   The name of the module to analyze.
   * @param array &$context
   *   The batch context array.
   */
  public function processModuleAnalysis($module_name, &$context) {
    try {
      // Initialize progress if not set
      if (!isset($context['sandbox']['progress'])) {
        $context['sandbox']['progress'] = 0;
        $context['sandbox']['current_module'] = 0;
        $context['sandbox']['max'] = 1;
        $context['results']['analyzed'] = [];
        $context['results']['errors'] = [];
      }

      // Skip if module doesn't exist
      if (!$this->moduleHandler->moduleExists($module_name)) {
        $context['results']['errors'][$module_name] = $this->t('Module @name does not exist.', [
          '@name' => $module_name,
        ]);
        return;
      }

      // Analyze the module
      $analysis = $this->projectAnalyzer->analyzeModule($module_name);
      
      // Search for patches
      $patches = $this->patchSearcher->findPatches($module_name, $analysis);
      
      // Store results
      $this->analysisTracker->saveAnalysis($module_name, $analysis, $patches);
      
      // Update progress
      $context['sandbox']['progress']++;
      $context['message'] = $this->t('Analyzed module: @name', ['@name' => $module_name]);
      $context['results']['analyzed'][] = $module_name;
    }
    catch (\Exception $e) {
      $context['results']['errors'][$module_name] = $e->getMessage();
    }
  }

  /**
   * Finish callback for the analysis batch.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The batch results array.
   * @param array $operations
   *   The batch operations array.
   */
  public function finishAnalysisBatch($success, $results, $operations) {
    if ($success) {
      if (!empty($results['errors'])) {
        foreach ($results['errors'] as $module => $error) {
          \Drupal::messenger()->addError(
            $this->t('Error analyzing @module: @error', [
              '@module' => $module,
              '@error' => $error,
            ])
          );
        }
      }
      
      $count = count($results['analyzed']);
      \Drupal::messenger()->addStatus($this->t('Successfully analyzed @count modules.', [
        '@count' => $count,
      ]));
    }
    else {
      \Drupal::messenger()->addError($this->t('An error occurred during module analysis.'));
    }
  }
}
