<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\State\StateInterface;

/**
 * Service for detecting the current environment type.
 */
class EnvironmentDetector {

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new EnvironmentDetector.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(StateInterface $state) {
    $this->state = $state;
  }

  /**
   * Determines if the current environment is a local development environment.
   *
   * @return bool
   *   TRUE if the environment is local development, FALSE otherwise.
   */
  public function isLocalDevelopment() {
    // Check for common local development environment indicators
    $indicators = [
      // DDEV
      'DDEV_APPROOT',
      'IS_DDEV_PROJECT',
      // Lando
      'LANDO',
      'LANDO_INFO',
      // Docker
      'DOCKER',
      'DOCKER_HOST',
      // General development
      'DEVELOPMENT',
      'DEV_MODE',
      // Drupal specific
      'DRUPAL_DEV_MODE',
    ];

    foreach ($indicators as $indicator) {
      if (!empty(getenv($indicator))) {
        return TRUE;
      }
    }

    // Check for common local hostnames
    $hostname = gethostname();
    $localHostPatterns = [
      '/^localhost/',
      '/\.local$/',
      '/\.test$/',
      '/\.ddev\.site$/',
      '/\.lndo\.site$/',
      '/\.docker\.localhost$/',
    ];

    foreach ($localHostPatterns as $pattern) {
      if (preg_match($pattern, $hostname)) {
        return TRUE;
      }
    }

    // Check if we're in a known local IP range
    $serverIp = $_SERVER['SERVER_ADDR'] ?? '';
    $localIpRanges = [
      '/^127\./',      // Loopback
      '/^10\./',       // Private network
      '/^172\.(1[6-9]|2[0-9]|3[0-1])\./', // Private network
      '/^192\.168\./', // Private network
    ];

    foreach ($localIpRanges as $range) {
      if (preg_match($range, $serverIp)) {
        return TRUE;
      }
    }

    // Allow manual override through state
    if ($this->state->get('ai_upgrade_assistant.force_local_environment')) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Forces the environment to be treated as local development.
   *
   * @param bool $isLocal
   *   Whether to force the environment as local (TRUE) or not (FALSE).
   */
  public function forceLocalEnvironment($isLocal = TRUE) {
    $this->state->set('ai_upgrade_assistant.force_local_environment', $isLocal);
  }

}
