<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service;

use Drupal\ai_upgrade_assistant\Service\OpenAIService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Prophecy\Argument;

/**
 * Unit tests for the OpenAI service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\OpenAIService
 */
class OpenAIServiceTest extends UnitTestCase {

  /**
   * The OpenAI service under test.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openAiService;

  /**
   * The HTTP client prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $httpClient;

  /**
   * The config factory prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $configFactory;

  /**
   * The logger factory prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $loggerFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create prophecies for dependencies
    $this->httpClient = $this->prophesize(ClientInterface::class);
    $this->configFactory = $this->prophesize(ConfigFactoryInterface::class);
    $this->loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);

    // Setup logger channel
    $loggerChannel = $this->prophesize(LoggerChannelInterface::class);
    $this->loggerFactory->get('ai_upgrade_assistant')
      ->willReturn($loggerChannel->reveal());

    // Setup config
    $config = $this->prophesize(ImmutableConfig::class);
    $config->get('openai_api_key')->willReturn('sk-validtestapikey123456789012345678901234567890');
    $this->configFactory->get('ai_upgrade_assistant.settings')
      ->willReturn($config->reveal());

    // Create service instance
    $this->openAiService = new OpenAIService(
      $this->httpClient->reveal(),
      $this->configFactory->reveal(),
      $this->loggerFactory->reveal()
    );
  }

  /**
   * Tests successful code analysis.
   *
   * @covers ::analyzeCode
   */
  public function testSuccessfulCodeAnalysis() {
    $code = '<?php function test() { return TRUE; }';
    $context = ['module' => 'test_module'];

    $mockResponse = [
      'choices' => [
        [
          'message' => [
            'content' => "SEVERITY: medium\nEXPLANATION: Test explanation\nSUGGESTIONS:\n1. First suggestion\n2. Second suggestion\nNEXT_STEPS: Test next steps",
          ],
        ],
      ],
    ];

    $this->httpClient->post('https://api.openai.com/v1/chat/completions', Argument::any())
      ->willReturn(new Response(200, [], json_encode($mockResponse)));

    $result = $this->openAiService->analyzeCode($code, $context);

    $this->assertEquals('medium', $result['severity']);
    $this->assertEquals('Test explanation', $result['explanation']);
    $this->assertEquals(['First suggestion', 'Second suggestion'], $result['suggestions']);
    $this->assertEquals('Test next steps', $result['next_steps']);
  }

  /**
   * Tests code analysis with rate limiting and retry.
   *
   * @covers ::analyzeCode
   * @covers ::makeApiCallWithRetry
   */
  public function testCodeAnalysisWithRateLimitRetry() {
    $code = '<?php function test() { return TRUE; }';

    // Mock rate limit response followed by success
    $rateLimitException = new RequestException(
      'Rate limit exceeded',
      new Request('POST', 'test'),
      new Response(429)
    );

    $mockResponse = [
      'choices' => [
        [
          'message' => [
            'content' => "SEVERITY: low\nEXPLANATION: Test\nSUGGESTIONS:\n1. Test\nNEXT_STEPS: Test",
          ],
        ],
      ],
    ];

    $this->httpClient->post('https://api.openai.com/v1/chat/completions', Argument::any())
      ->willThrow($rateLimitException)
      ->shouldBeCalledTimes(1);

    $this->httpClient->post('https://api.openai.com/v1/chat/completions', Argument::any())
      ->willReturn(new Response(200, [], json_encode($mockResponse)))
      ->shouldBeCalledTimes(1);

    $result = $this->openAiService->analyzeCode($code);
    $this->assertEquals('low', $result['severity']);
  }

  /**
   * Tests code analysis in test mode.
   *
   * @covers ::analyzeCode
   * @covers ::getMockAnalysisResults
   */
  public function testCodeAnalysisInTestMode() {
    $this->openAiService->setTestMode(TRUE);
    $result = $this->openAiService->analyzeCode('test code');

    $this->assertArrayHasKey('severity', $result);
    $this->assertArrayHasKey('explanation', $result);
    $this->assertArrayHasKey('suggestions', $result);
    $this->assertArrayHasKey('next_steps', $result);
    $this->assertIsArray($result['suggestions']);
  }

  /**
   * Tests invalid API key format.
   *
   * @covers ::analyzeCode
   * @covers ::isValidApiKeyFormat
   */
  public function testInvalidApiKeyFormat() {
    $config = $this->prophesize(ImmutableConfig::class);
    $config->get('openai_api_key')->willReturn('invalid-key');
    $this->configFactory->get('ai_upgrade_assistant.settings')
      ->willReturn($config->reveal());

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Invalid OpenAI API key format.');

    $this->openAiService->analyzeCode('test code');
  }

  /**
   * Tests empty API response handling.
   *
   * @covers ::analyzeCode
   * @covers ::parseAnalysisResponse
   */
  public function testEmptyApiResponse() {
    $mockResponse = ['choices' => [['message' => ['content' => '']]]];

    $this->httpClient->post('https://api.openai.com/v1/chat/completions', Argument::any())
      ->willReturn(new Response(200, [], json_encode($mockResponse)));

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Empty response from OpenAI');

    $this->openAiService->analyzeCode('test code');
  }

  /**
   * Tests response parsing with partial content.
   *
   * @covers ::parseAnalysisResponse
   * @covers ::extractSeverity
   * @covers ::extractExplanation
   * @covers ::extractSuggestions
   * @covers ::extractNextStep
   */
  public function testResponseParsingWithPartialContent() {
    $code = '<?php function test() { return TRUE; }';
    
    $mockResponse = [
      'choices' => [
        [
          'message' => [
            'content' => "Some unstructured content without proper sections",
          ],
        ],
      ],
    ];

    $this->httpClient->post('https://api.openai.com/v1/chat/completions', Argument::any())
      ->willReturn(new Response(200, [], json_encode($mockResponse)));

    $result = $this->openAiService->analyzeCode($code);

    // Check default values for missing sections
    $this->assertEquals('medium', $result['severity']);
    $this->assertEquals('', $result['explanation']);
    $this->assertEquals([], $result['suggestions']);
    $this->assertEquals('', $result['next_steps']);
  }

}
