<?php

namespace Drupal\ai_upgrade_assistant\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\user\EntityOwnerTrait;
use Drupal\user\EntityOwnerInterface;

/**
 * Defines the Upgrade Analysis entity.
 *
 * @ContentEntityType(
 *   id = "upgrade_analysis",
 *   label = @Translation("Upgrade Analysis"),
 *   base_table = "upgrade_analysis",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *     "created" = "created",
 *     "changed" = "changed",
 *   },
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\ai_upgrade_assistant\Entity\Controller\UpgradeAnalysisListBuilder",
 *     "form" = {
 *       "default" = "Drupal\ai_upgrade_assistant\Form\UpgradeAnalysisForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "access" = "Drupal\ai_upgrade_assistant\Entity\AccessControl\UpgradeAnalysisAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   links = {
 *     "canonical" = "/admin/reports/upgrade-analysis/{upgrade_analysis}",
 *     "add-form" = "/admin/reports/upgrade-analysis/add",
 *     "edit-form" = "/admin/reports/upgrade-analysis/{upgrade_analysis}/edit",
 *     "delete-form" = "/admin/reports/upgrade-analysis/{upgrade_analysis}/delete",
 *     "collection" = "/admin/reports/upgrade-analysis",
 *   },
 * )
 */
class UpgradeAnalysis extends ContentEntityBase implements EntityOwnerInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['module_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Module Name'))
      ->setDescription(t('The name of the module being analyzed.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['current_version'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Current Version'))
      ->setDescription(t('The current version of the module.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['target_version'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Target Version'))
      ->setDescription(t('The target version for upgrade.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['analysis_data'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Analysis Data'))
      ->setDescription(t('The detailed analysis data in JSON format.'))
      ->setRequired(FALSE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Status'))
      ->setDescription(t('The status of the upgrade analysis.'))
      ->setRequired(TRUE)
      ->setSettings([
        'allowed_values' => [
          'pending' => 'Pending',
          'in_progress' => 'In Progress',
          'completed' => 'Completed',
          'failed' => 'Failed',
        ],
      ])
      ->setDefaultValue('pending')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'list_default',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the analysis was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the analysis was last edited.'));

    return $fields;
  }

}
