<?php

namespace Drupal\ai_upgrade_assistant\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Organization settings form for AI Upgrade Assistant.
 */
class OrganizationSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_upgrade_assistant.organization_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_upgrade_assistant_organization_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_upgrade_assistant.organization_settings');

    $form['organization'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Organization Profile'),
      '#description' => $this->t('Configure your organization profile for HuggingFace integration.'),
    ];

    $form['organization']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Organization Username'),
      '#default_value' => $config->get('username') ?: 'thronedigital',
      '#required' => TRUE,
      '#description' => $this->t('Your organization username on HuggingFace.'),
    ];

    $form['organization']['full_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Organization Full Name'),
      '#default_value' => $config->get('full_name'),
      '#required' => TRUE,
      '#description' => $this->t('The full name of your organization.'),
    ];

    $form['organization']['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Organization Type'),
      '#options' => [
        'company' => $this->t('Company'),
        'academic' => $this->t('Academic Institution'),
        'nonprofit' => $this->t('Non-Profit'),
        'government' => $this->t('Government'),
        'individual' => $this->t('Individual'),
        'other' => $this->t('Other'),
      ],
      '#default_value' => $config->get('type') ?: 'company',
      '#required' => TRUE,
    ];

    $form['organization']['homepage'] = [
      '#type' => 'url',
      '#title' => $this->t('Homepage'),
      '#default_value' => $config->get('homepage'),
      '#description' => $this->t('Your organization website (optional).'),
    ];

    $form['organization']['github_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GitHub Username'),
      '#default_value' => $config->get('github_username'),
      '#description' => $this->t('Your GitHub username (optional).'),
    ];

    $form['organization']['twitter_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Twitter Username'),
      '#default_value' => $config->get('twitter_username'),
      '#description' => $this->t('Your Twitter username (optional).'),
    ];

    $form['organization']['ai_interests'] = [
      '#type' => 'textarea',
      '#title' => $this->t('AI & ML Interests'),
      '#default_value' => $config->get('ai_interests'),
      '#description' => $this->t('Describe your organization\'s interests in AI and Machine Learning (optional).'),
      '#rows' => 3,
    ];

    $form['organization']['logo'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Logo'),
      '#upload_location' => 'public://ai_upgrade_assistant/logos',
      '#upload_validators' => [
        'file_validate_extensions' => ['png jpg jpeg svg'],
        'file_validate_size' => [2 * 1024 * 1024], // 2MB max
      ],
      '#default_value' => $config->get('logo'),
      '#description' => $this->t('Upload your organization logo (optional, max 2MB, PNG/JPG/SVG).'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_upgrade_assistant.organization_settings');
    
    $fields = [
      'username',
      'full_name',
      'type',
      'homepage',
      'github_username',
      'twitter_username',
      'ai_interests',
      'logo',
    ];

    foreach ($fields as $field) {
      $config->set($field, $form_state->getValue($field));
    }

    $config->save();

    // Update the HuggingFace organization profile
    \Drupal::service('ai_upgrade_assistant.huggingface')->updateOrganizationProfile($config->get());

    parent::submitForm($form, $form_state);
  }

}
