<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for managing achievements and character progression.
 */
class AchievementService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Epic titles for each level range.
   *
   * @var array
   */
  protected $epicTitles = [
    1 => [
      'Backend Squire',
      'Code Apprentice',
      'Module Initiate',
      'Patch Novice',
    ],
    5 => [
      'Function Slayer',
      'Deprecated Hunter',
      'Pattern Seeker',
      'Bug Vanquisher',
    ],
    10 => [
      'Code Paladin',
      'Module Warlock',
      'Patch Ranger',
      'Integration Barbarian',
    ],
    15 => [
      'Archmage of Updates',
      'Legendary Debugger',
      'Supreme Pattern Lord',
      'Immortal Integrator',
    ],
    20 => [
      'God-Emperor of Code',
      'The Omnipotent Upgrader',
      'Master of All Patches',
      'The Undying Integrator',
    ],
  ];

  /**
   * Epic achievements with hilarious descriptions.
   *
   * @var array
   */
  protected $epicAchievements = [
    'first_upgrade' => [
      'name' => 'First Blood!',
      'description' => "You've drawn first blood against deprecated code. The old functions cry out in terror!",
      'xp' => 100,
    ],
    'mass_upgrade' => [
      'name' => 'UPGRADE-AKILL!',
      'description' => "By the gods! You've upgraded 5 modules in under an hour! Someone stop this madness!",
      'xp' => 500,
    ],
    'pattern_master' => [
      'name' => 'Pattern Whisperer',
      'description' => "You speak the ancient language of code patterns. The AI bows before your wisdom!",
      'xp' => 300,
    ],
    'bug_slayer' => [
      'name' => 'Legendary Bug Slayer',
      'description' => "That wasn't a bug, that was a feature... until you DESTROYED it!",
      'xp' => 250,
    ],
    'community_hero' => [
      'name' => 'Hero of the People',
      'description' => "Your contributions have saved countless developers from the depths of dependency hell!",
      'xp' => 400,
    ],
    'integration_wizard' => [
      'name' => 'Integration Archmage',
      'description' => "You've mastered the arcane art of making everything work together... somehow!",
      'xp' => 450,
    ],
    'speed_demon' => [
      'name' => 'UPGRADE-LOCITY!',
      'description' => "You're upgrading modules faster than a caffeinated squirrel on a sugar rush!",
      'xp' => 350,
    ],
    'patch_savior' => [
      'name' => 'Patch from the Ashes',
      'description' => "Like a phoenix, you've brought life to a dying module. It's ALIVE!",
      'xp' => 300,
    ],
  ];

  /**
   * Epic failure titles (because failing is fun too!).
   *
   * @var array
   */
  protected $epicFailTitles = [
    'Breaker of Builds',
    'Chaos Agent',
    'Master of Disasters',
    'The Production Crasher',
    'Dependency Hell Tourist',
    'Bug Creator Extraordinaire',
    'Loop Lord of Infinity',
    'Exception Enthusiast',
  ];

  /**
   * Constructs a new AchievementService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    AccountInterface $current_user,
    Connection $database
  ) {
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
    $this->currentUser = $current_user;
    $this->database = $database;
  }

  /**
   * Awards experience points with epic fanfare!
   */
  public function awardExperience($uid, $amount, $reason) {
    try {
      $current_xp = $this->getCurrentXP($uid);
      $new_xp = $current_xp + $amount;
      
      // Update XP in database
      $this->database->merge('ai_upgrade_achievements')
        ->key(['uid' => $uid])
        ->fields([
          'experience' => $new_xp,
          'last_updated' => time(),
        ])
        ->execute();

      // Check for level up
      $old_level = $this->calculateLevel($current_xp);
      $new_level = $this->calculateLevel($new_xp);

      if ($new_level > $old_level) {
        $this->levelUp($uid, $new_level);
      }

      // Epic messages based on XP gained
      $messages = [
        "By the power of clean code, you gain {$amount} XP!",
        "The deprecated gods smile upon you! +{$amount} XP",
        "GLORIOUS UPGRADE! You've earned {$amount} XP!",
        "Your code-fu grows stronger! +{$amount} XP",
        "The ancients approve of your changes! {$amount} XP granted!",
      ];

      $this->messenger->addStatus($messages[array_rand($messages)]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Failed to award XP: @error', ['@error' => $e->getMessage()]
      );
      return FALSE;
    }
  }

  /**
   * Handles the epic moment of leveling up!
   */
  protected function levelUp($uid, $new_level) {
    $title = $this->getEpicTitle($new_level);
    
    $messages = [
      "🎉 LEVEL UP! You are now a mighty level {$new_level} {$title}!",
      "⚔️ The code trembles before you, Level {$new_level} {$title}!",
      "🏆 BEHOLD! You have ascended to Level {$new_level} {$title}!",
      "⚡ Your power grows! Welcome to Level {$new_level}, {$title}!",
      "🌟 The stars align! You've become a Level {$new_level} {$title}!",
    ];

    $this->messenger->addStatus($messages[array_rand($messages)]);

    // Special level-up rewards
    if ($new_level % 5 === 0) {
      $this->grantSpecialReward($uid, $new_level);
    }
  }

  /**
   * Grants special rewards for milestone levels.
   */
  protected function grantSpecialReward($uid, $level) {
    $rewards = [
      5 => "You've unlocked the mystical 'Ctrl+Z' enchantment!",
      10 => "The legendary 'Coffee Break' power is now yours!",
      15 => "You've gained the ability to summon Stack Overflow at will!",
      20 => "GODMODE UNLOCKED: You can now read error messages... sometimes.",
    ];

    if (isset($rewards[$level])) {
      $this->messenger->addStatus($rewards[$level]);
    }
  }

  /**
   * Gets an epic title based on level.
   */
  protected function getEpicTitle($level) {
    foreach (array_reverse(array_keys($this->epicTitles)) as $threshold) {
      if ($level >= $threshold) {
        $titles = $this->epicTitles[$threshold];
        return $titles[array_rand($titles)];
      }
    }
    return $this->epicTitles[1][0];
  }

  /**
   * Calculates level based on XP.
   */
  protected function calculateLevel($xp) {
    // Each level requires 20% more XP than the last
    $level = 1;
    $xp_required = 100;
    $total_xp_required = 0;

    while ($xp >= ($total_xp_required + $xp_required)) {
      $total_xp_required += $xp_required;
      $xp_required = floor($xp_required * 1.2);
      $level++;
    }

    return min($level, 20); // Cap at level 20 (for now!)
  }

  /**
   * Gets current XP for a user.
   */
  protected function getCurrentXP($uid) {
    $result = $this->database->select('ai_upgrade_achievements', 'a')
      ->fields('a', ['experience'])
      ->condition('uid', $uid)
      ->execute()
      ->fetchField();

    return $result ? (int) $result : 0;
  }

  /**
   * Awards an epic achievement with maximum drama!
   */
  public function awardAchievement($uid, $achievement_key) {
    if (!isset($this->epicAchievements[$achievement_key])) {
      return FALSE;
    }

    $achievement = $this->epicAchievements[$achievement_key];
    
    // Record the achievement
    $this->database->insert('ai_upgrade_achievements')
      ->fields([
        'uid' => $uid,
        'achievement_key' => $achievement_key,
        'awarded' => time(),
      ])
      ->execute();

    // Award XP
    $this->awardExperience($uid, $achievement['xp'], $achievement['name']);

    // Epic announcement
    $messages = [
      "🎊 ACHIEVEMENT UNLOCKED: {$achievement['name']}!",
      "⚡ INCREDIBLE! You've earned: {$achievement['name']}!",
      "🏆 VICTORY! Achievement Unlocked: {$achievement['name']}!",
      "✨ BY THE GODS! You've achieved: {$achievement['name']}!",
    ];

    $this->messenger->addStatus($messages[array_rand($messages)]);
    $this->messenger->addStatus($achievement['description']);

    return TRUE;
  }

  /**
   * Sometimes failure is just as epic as success!
   */
  public function recordEpicFail($uid, $fail_type) {
    $title = $this->epicFailTitles[array_rand($this->epicFailTitles)];
    
    $messages = [
      "Congratulations! You're now a certified '{$title}'!",
      "Achievement Unlocked: Most Creative Way to Break Everything!",
      "Task Failed Successfully! You've become a '{$title}'!",
      "The chaos gods are pleased with your offering!",
    ];

    $this->messenger->addStatus($messages[array_rand($messages)]);
  }
}
