<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for checking environment requirements.
 */
class EnvironmentCheckerService {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs an EnvironmentCheckerService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    StateInterface $state
  ) {
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->state = $state;
  }

  /**
   * Runs environment checks.
   *
   * @return array
   *   An array of check results.
   */
  public function runChecks() {
    $checks = [];

    // Check PHP version.
    $checks[] = [
      '#type' => 'container',
      'status' => [
        '#type' => 'markup',
        '#markup' => version_compare(PHP_VERSION, '8.1.0', '>=') ? 'success' : 'error',
      ],
      'message' => [
        '#type' => 'markup',
        '#markup' => t('PHP version @version', ['@version' => PHP_VERSION]),
      ],
    ];

    // Check if required modules are enabled.
    $required_modules = ['upgrade_status'];
    $missing_modules = [];
    foreach ($required_modules as $module) {
      if (!$this->moduleHandler->moduleExists($module)) {
        $missing_modules[] = $module;
      }
    }

    $checks[] = [
      '#type' => 'container',
      'status' => [
        '#type' => 'markup',
        '#markup' => empty($missing_modules) ? 'success' : 'error',
      ],
      'message' => [
        '#type' => 'markup',
        '#markup' => empty($missing_modules) 
          ? t('Required modules installed')
          : t('Missing required modules: @modules', ['@modules' => implode(', ', $missing_modules)]),
      ],
    ];

    // Check HuggingFace API key.
    $api_key = $this->state->get('ai_upgrade_assistant.huggingface_api_key');
    $checks[] = [
      '#type' => 'container',
      'status' => [
        '#type' => 'markup',
        '#markup' => !empty($api_key) ? 'success' : 'error',
      ],
      'message' => [
        '#type' => 'markup',
        '#markup' => !empty($api_key) 
          ? t('HuggingFace API key is properly configured.')
          : t('Please configure your HuggingFace API key in the module settings.'),
      ],
    ];

    return $checks;
  }

}
