<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Manages AI service subscriptions and usage.
 */
class SubscriptionManager {

  use DependencySerializationTrait;

  /**
   * Subscription tiers.
   */
  const TIER_COMMUNITY = 'community';
  const TIER_PRO = 'pro';
  const TIER_ENTERPRISE = 'enterprise';

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new SubscriptionManager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
  }

  /**
   * Gets the current subscription tier.
   *
   * @return string
   *   The subscription tier (community, pro, or enterprise).
   */
  public function getCurrentTier() {
    return $this->state->get('ai_upgrade_assistant.subscription_tier', self::TIER_COMMUNITY);
  }

  /**
   * Checks if a feature is available in the current tier.
   *
   * @param string $feature
   *   The feature to check.
   *
   * @return bool
   *   TRUE if the feature is available, FALSE otherwise.
   */
  public function hasFeature($feature) {
    $tier = $this->getCurrentTier();
    $features = $this->getFeaturesByTier($tier);
    return in_array($feature, $features);
  }

  /**
   * Gets available features for a subscription tier.
   *
   * @param string $tier
   *   The subscription tier.
   *
   * @return array
   *   List of available features.
   */
  protected function getFeaturesByTier($tier) {
    $features = [
      self::TIER_COMMUNITY => [
        'pattern_database_access',
        'basic_code_analysis',
        'community_contributions',
      ],
      self::TIER_PRO => [
        'pattern_database_access',
        'basic_code_analysis',
        'community_contributions',
        'ai_code_analysis',
        'custom_patterns',
        'priority_support',
        'huggingface_integration',
        'openai_integration',
      ],
      self::TIER_ENTERPRISE => [
        'pattern_database_access',
        'basic_code_analysis',
        'community_contributions',
        'ai_code_analysis',
        'custom_patterns',
        'priority_support',
        'huggingface_integration',
        'openai_integration',
        'private_patterns',
        'custom_model_training',
        'sla_support',
      ],
    ];

    return $features[$tier] ?? $features[self::TIER_COMMUNITY];
  }

  /**
   * Gets usage statistics for the current period.
   *
   * @return array
   *   Usage statistics.
   */
  public function getUsageStats() {
    $stats = $this->state->get('ai_upgrade_assistant.usage_stats', []);
    return [
      'api_calls' => $stats['api_calls'] ?? 0,
      'patterns_created' => $stats['patterns_created'] ?? 0,
      'patterns_used' => $stats['patterns_used'] ?? 0,
      'period_start' => $stats['period_start'] ?? time(),
      'period_end' => $stats['period_end'] ?? strtotime('+1 month'),
    ];
  }

  /**
   * Tracks API usage.
   *
   * @param string $type
   *   The type of API call.
   * @param int $tokens
   *   Number of tokens used.
   */
  public function trackUsage($type, $tokens = 0) {
    $stats = $this->getUsageStats();
    $stats['api_calls']++;
    $stats['tokens_used'] = ($stats['tokens_used'] ?? 0) + $tokens;
    
    // Reset stats if period has ended
    if (time() > $stats['period_end']) {
      $stats = [
        'api_calls' => 1,
        'tokens_used' => $tokens,
        'period_start' => time(),
        'period_end' => strtotime('+1 month'),
      ];
    }

    $this->state->set('ai_upgrade_assistant.usage_stats', $stats);
  }

  /**
   * Checks if usage limits have been exceeded.
   *
   * @return bool
   *   TRUE if limits exceeded, FALSE otherwise.
   */
  public function hasExceededLimits() {
    $tier = $this->getCurrentTier();
    $stats = $this->getUsageStats();
    
    $limits = [
      self::TIER_COMMUNITY => [
        'api_calls' => 100,
        'patterns_created' => 10,
      ],
      self::TIER_PRO => [
        'api_calls' => 1000,
        'patterns_created' => 100,
      ],
      self::TIER_ENTERPRISE => [
        'api_calls' => PHP_INT_MAX,
        'patterns_created' => PHP_INT_MAX,
      ],
    ];

    $tier_limits = $limits[$tier] ?? $limits[self::TIER_COMMUNITY];
    return $stats['api_calls'] > $tier_limits['api_calls'];
  }

}
