<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Service for handling the automated upgrade process.
 */
class UpgradeService {

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openAiService;

  /**
   * The module analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ModuleAnalyzerService
   */
  protected $moduleAnalyzer;

  /**
   * The patch generator service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchGenerator
   */
  protected $patchGenerator;

  /**
   * The patch validator service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchValidator
   */
  protected $patchValidator;

  /**
   * The rollback manager service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\RollbackManager
   */
  protected $rollbackManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new UpgradeService.
   */
  public function __construct(
    OpenAIService $openai_service,
    ModuleAnalyzerService $module_analyzer,
    PatchGenerator $patch_generator,
    PatchValidator $patch_validator,
    RollbackManager $rollback_manager,
    ModuleHandlerInterface $module_handler,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system
  ) {
    $this->openAiService = $openai_service;
    $this->moduleAnalyzer = $module_analyzer;
    $this->patchGenerator = $patch_generator;
    $this->patchValidator = $patch_validator;
    $this->rollbackManager = $rollback_manager;
    $this->moduleHandler = $module_handler;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
    $this->fileSystem = $file_system;
  }

  /**
   * Starts the automated upgrade process for a module.
   *
   * @param string $module_name
   *   The name of the module to upgrade.
   *
   * @return array
   *   Status information about the upgrade process.
   */
  public function startUpgrade($module_name) {
    try {
      // 1. Create backup
      $backup_path = $this->rollbackManager->createBackup($module_name);
      
      // 2. Analyze module code
      $analysis = $this->moduleAnalyzer->analyzeModule($module_name);
      
      // 3. Generate patches for each issue
      $patches = [];
      foreach ($analysis['issues'] as $issue) {
        $patch = $this->patchGenerator->generatePatch(
          $issue['file'],
          $issue['code'],
          $issue['description']
        );
        
        // Validate patch
        if ($this->patchValidator->validatePatch($patch)) {
          $patches[] = $patch;
        }
      }
      
      // 4. Apply patches
      $applied = [];
      foreach ($patches as $patch) {
        try {
          $this->patchGenerator->applyPatch($patch);
          $applied[] = $patch;
        }
        catch (\Exception $e) {
          // Log failure and continue with next patch
          $this->loggerFactory->get('ai_upgrade_assistant')->error(
            'Failed to apply patch: @error', ['@error' => $e->getMessage()]
          );
        }
      }
      
      // 5. Run tests if available
      $test_results = $this->runTests($module_name);
      
      // 6. If tests fail, rollback
      if (!empty($test_results['failures'])) {
        $this->rollbackManager->rollback($module_name, $backup_path);
        throw new \Exception('Tests failed after upgrade. Changes have been rolled back.');
      }
      
      // 7. Update module status
      $this->state->set(
        "ai_upgrade_assistant.module_status.$module_name",
        [
          'status' => 'upgraded',
          'timestamp' => time(),
          'patches_applied' => count($applied),
          'test_results' => $test_results,
        ]
      );
      
      return [
        'status' => 'success',
        'patches_applied' => count($applied),
        'test_results' => $test_results,
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Upgrade failed for module @module: @error',
        ['@module' => $module_name, '@error' => $e->getMessage()]
      );
      
      // Ensure rollback on any error
      if (!empty($backup_path)) {
        $this->rollbackManager->rollback($module_name, $backup_path);
      }
      
      throw $e;
    }
  }

  /**
   * Runs automated tests for a module.
   *
   * @param string $module_name
   *   The name of the module.
   *
   * @return array
   *   Test results.
   */
  protected function runTests($module_name) {
    // Get test classes for this module
    $test_classes = $this->moduleHandler->invokeAll('ai_upgrade_assistant_tests', [$module_name]);
    
    $results = [
      'total' => 0,
      'passed' => 0,
      'failures' => [],
    ];
    
    foreach ($test_classes as $test_class) {
      try {
        // Run PHPUnit tests
        $test_results = TestRunnerService::runTest($test_class);
        $results['total'] += $test_results['total'];
        $results['passed'] += $test_results['passed'];
        if (!empty($test_results['failures'])) {
          $results['failures'] = array_merge($results['failures'], $test_results['failures']);
        }
      }
      catch (\Exception $e) {
        $results['failures'][] = [
          'class' => $test_class,
          'error' => $e->getMessage(),
        ];
      }
    }
    
    return $results;
  }

}
